<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Produk;
use Illuminate\Support\Facades\Storage;
use App\Models\User;

class ProdukController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // Mulai query dasar
        $query = Produk::query();

        // Jika ada parameter pencarian 'q', filter berdasarkan nama produk
        if ($request->filled('q')) {
            $q = $request->q;
            $query->where('nama', 'like', "%{$q}%");
        }

        // Filter berdasarkan stok: min_stok dan max_stok (dashboard mengirim max_stok=5 untuk stok menipis)
        $min = $request->input('min_stok');
        $max = $request->input('max_stok');
        if ($min !== null && $min !== '' && $max !== null && $max !== '') {
            $query->whereBetween('stok', [(int)$min, (int)$max]);
        } else {
            if ($min !== null && $min !== '') {
                $query->where('stok', '>=', (int)$min);
            }
            if ($max !== null && $max !== '') {
                $query->where('stok', '<=', (int)$max);
            }
        }

        // Paginate dan sertakan query string agar parameter tetap ada di pagination links
        $produk = $query->paginate(10)->appends($request->only('q', 'min_stok', 'max_stok'));

        return view('produk.index', compact('produk'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('produk.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'stok' => 'required|integer',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);
        $data = $request->only('nama', 'stok');
        if ($request->hasFile('foto')) {
            $data['foto'] = $request->file('foto')->store('produk', 'public');
        }
        Produk::create($data);
        return redirect()->route('produk.index')->with('success', 'Produk berhasil ditambahkan!');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $produk = Produk::findOrFail($id);
        return view('produk.show', compact('produk'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $produk = Produk::findOrFail($id);
        return view('produk.edit', compact('produk'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'nama' => 'required',
            'stok' => 'required|integer',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);
        
        $produk = Produk::findOrFail($id);
        $data = $request->only('nama', 'stok');
        
        if ($request->hasFile('foto')) {
            // Hapus foto lama jika ada
            if ($produk->foto) {
                Storage::disk('public')->delete($produk->foto);
            }
            $data['foto'] = $request->file('foto')->store('produk', 'public');
        }
        
        $produk->update($data);
        return redirect()->route('produk.index')->with('success', 'Produk berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $produk = Produk::findOrFail($id);
        
        // Hapus foto jika ada
        if ($produk->foto) {
            Storage::disk('public')->delete($produk->foto);
        }
        
        $produk->delete();
        return redirect()->route('produk.index')->with('success', 'Produk berhasil dihapus!');
    }
}
