<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\AktivitasModel;

class Aktivitas extends BaseController
{
    protected $aktivitasModel;
    protected $validation;

    public function __construct()
    {
        $this->aktivitasModel = new AktivitasModel();
        $this->validation = \Config\Services::validation();
    }

    /**
     * Display list of activities
     */
    public function index()
    {
        $data = [
            'title' => 'Data Aktivitas Harian',
            'aktivitas' => $this->aktivitasModel->orderBy('tanggal', 'DESC')->orderBy('jam', 'DESC')->findAll(),
        ];

        return view('admin/aktivitas/index', $data);
    }

    /**
     * Show create form
     */
    public function create()
    {
        $data = [
            'title' => 'Tambah Aktivitas Harian',
        ];

        return view('admin/aktivitas/create', $data);
    }

    /**
     * Store new activity
     */
    public function store()
    {
        $rules = [
            'tanggal' => 'required|valid_date',
            'jam' => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png,image/gif]',
            'video' => 'max_size[video,10240]|mime_in[video,video/mp4,video/avi,video/mov,video/webm]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi'),
        ];

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'assets/uploads/aktivitas', $newName);
            $data['foto'] = $newName;
        }

        // Handle video upload
        $video = $this->request->getFile('video');
        if ($video && $video->isValid() && !$video->hasMoved()) {
            $newName = $video->getRandomName();
            $video->move(FCPATH . 'assets/uploads/aktivitas', $newName);
            $data['video'] = $newName;
        }

        $this->aktivitasModel->insert($data);

        return redirect()->to('/admin/aktivitas')->with('success', 'Aktivitas berhasil ditambahkan!');
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            return redirect()->to('/admin/aktivitas')->with('error', 'Data tidak ditemukan.');
        }

        $data = [
            'title' => 'Edit Aktivitas Harian',
            'aktivitas' => $aktivitas,
        ];

        return view('admin/aktivitas/edit', $data);
    }

    /**
     * Update activity
     */
    public function update($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            return redirect()->to('/admin/aktivitas')->with('error', 'Data tidak ditemukan.');
        }

        $rules = [
            'tanggal' => 'required|valid_date',
            'jam' => 'required',
            'nama_aktivitas' => 'required|min_length[3]|max_length[255]',
            'foto' => 'max_size[foto,2048]|is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png,image/gif]',
            'video' => 'max_size[video,10240]|mime_in[video,video/mp4,video/avi,video/mov,video/webm]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi'),
        ];

        // Handle delete foto checkbox
        if ($this->request->getPost('hapus_foto')) {
            if ($aktivitas['foto'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto'])) {
                unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto']);
            }
            $data['foto'] = null;
        }

        // Handle delete video checkbox
        if ($this->request->getPost('hapus_video')) {
            if ($aktivitas['video'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video'])) {
                unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video']);
            }
            $data['video'] = null;
        }

        // Handle foto upload
        $foto = $this->request->getFile('foto');
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            // Delete old foto
            if ($aktivitas['foto'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto'])) {
                unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto']);
            }
            $newName = $foto->getRandomName();
            $foto->move(FCPATH . 'assets/uploads/aktivitas', $newName);
            $data['foto'] = $newName;
        }

        // Handle video upload
        $video = $this->request->getFile('video');
        if ($video && $video->isValid() && !$video->hasMoved()) {
            // Delete old video
            if ($aktivitas['video'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video'])) {
                unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video']);
            }
            $newName = $video->getRandomName();
            $video->move(FCPATH . 'assets/uploads/aktivitas', $newName);
            $data['video'] = $newName;
        }

        $this->aktivitasModel->update($id, $data);

        return redirect()->to('/admin/aktivitas')->with('success', 'Aktivitas berhasil diperbarui!');
    }

    /**
     * Delete activity
     */
    public function delete($id)
    {
        $aktivitas = $this->aktivitasModel->find($id);

        if (!$aktivitas) {
            return redirect()->to('/admin/aktivitas')->with('error', 'Data tidak ditemukan.');
        }

        // Delete associated files
        if ($aktivitas['foto'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto'])) {
            unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['foto']);
        }
        if ($aktivitas['video'] && file_exists(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video'])) {
            unlink(FCPATH . 'assets/uploads/aktivitas/' . $aktivitas['video']);
        }

        $this->aktivitasModel->delete($id);

        return redirect()->to('/admin/aktivitas')->with('success', 'Aktivitas berhasil dihapus!');
    }
}
