<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\ActivityModel;

class Activity extends BaseController
{
    protected $activityModel;

    public function __construct()
    {
        $this->activityModel = new ActivityModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search');
        $sort = $this->request->getGet('sort') ?? 'id';
        $order = $this->request->getGet('order') ?? 'DESC';

        $model = $this->activityModel;

        if ($search) {
            $model->groupStart()
                ->like('nama_aktivitas', $search)
                ->orLike('deskripsi', $search)
                ->orLike('kategori', $search)
                ->groupEnd();
        }

        $data = [
            'title' => 'Aktivitas Harian',
            'activities' => $model->orderBy($sort, $order)->paginate(10),
            'pager' => $model->pager,
            'search' => $search,
            'sort' => $sort,
            'order' => $order
        ];
        return view('admin/activity/index', $data);
    }

    public function create()
    {
        $data = ['title' => 'Tambah Aktivitas'];
        return view('admin/activity/create', $data);
    }

    public function store()
    {
        if (!$this->validate([
            'tanggal' => 'required',
            'jam' => 'required',
            'nama_aktivitas' => 'required',
            'foto' => 'is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png,image/webp]|max_size[foto,2048]',
            'video' => 'mime_in[video,video/mp4,video/webm]|max_size[video,10240]',
        ])) {
            return redirect()->to('/admin/activity/create')->withInput()->with('errors', $this->validator->getErrors());
        }

        $foto = $this->request->getFile('foto');
        $fotoName = null;
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $fotoName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/photos', $fotoName);
        }

        $video = $this->request->getFile('video');
        $videoName = null;
        if ($video && $video->isValid() && !$video->hasMoved()) {
            $videoName = $video->getRandomName();
            $video->move(ROOTPATH . 'public/uploads/videos', $videoName);
        }

        $this->activityModel->save([
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'kategori' => $this->request->getPost('kategori'),
            'foto' => $fotoName,
            'video' => $videoName,
        ]);

        return redirect()->to('/admin/activity')->with('message', 'Aktivitas berhasil disimpan');
    }

    public function edit($id)
    {
        $data = [
            'title' => 'Edit Aktivitas',
            'activity' => $this->activityModel->find($id)
        ];
        return view('admin/activity/edit', $data);
    }

    public function update($id)
    {
        if (!$this->validate([
            'foto' => 'is_image[foto]|mime_in[foto,image/jpg,image/jpeg,image/png,image/webp]|max_size[foto,2048]',
            'video' => 'mime_in[video,video/mp4,video/webm]|max_size[video,10240]',
        ])) {
            return redirect()->to('/admin/activity/edit/' . $id)->withInput()->with('errors', $this->validator->getErrors());
        }

        $activity = $this->activityModel->find($id);

        $foto = $this->request->getFile('foto');
        $fotoName = $activity['foto'];
        if ($foto && $foto->isValid() && !$foto->hasMoved()) {
            $fotoName = $foto->getRandomName();
            $foto->move(ROOTPATH . 'public/uploads/photos', $fotoName);
            // Optional: delete old file
        }

        $video = $this->request->getFile('video');
        $videoName = $activity['video'];
        if ($video && $video->isValid() && !$video->hasMoved()) {
            $videoName = $video->getRandomName();
            $video->move(ROOTPATH . 'public/uploads/videos', $videoName);
        }

        $this->activityModel->update($id, [
            'tanggal' => $this->request->getPost('tanggal'),
            'jam' => $this->request->getPost('jam'),
            'nama_aktivitas' => $this->request->getPost('nama_aktivitas'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'kategori' => $this->request->getPost('kategori'),
            'foto' => $fotoName,
            'video' => $videoName,
        ]);

        return redirect()->to('/admin/activity')->with('message', 'Aktivitas berhasil diperbarui');
    }

    public function delete($id)
    {
        $this->activityModel->delete($id);
        return redirect()->to('/admin/activity')->with('message', 'Aktivitas berhasil dihapus');
    }
}
