<?php

declare(strict_types=1);

/**
 * This file is part of CodeIgniter 4 framework.
 *
 * (c) CodeIgniter Foundation <admin@codeigniter.com>
 *
 * For the full copyright and license information, please view
 * the LICENSE file that was distributed with this source code.
 */

namespace CodeIgniter\Filters;

use CodeIgniter\Config\BaseConfig;
use CodeIgniter\Config\Filters as FiltersConfig;
use CodeIgniter\Filters\Exceptions\FilterException;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Config\Filters as AppFiltersConfig;

/**
 * Filters Class
 *
 * Filters allow you to run tasks before and/or after a controller
 * is executed. During before filters, the request can be modified,
 * and actions taken based on the request, while after filters can
 * act on or modify the response itself before it is sent to the client.
 */
class Filters
{
    /**
     * The original config file
     *
     * @var FiltersConfig
     */
    protected $config;

    /**
     * The active IncomingRequest or CLIRequest
     *
     * @var RequestInterface
     */
    protected $request;

    /**
     * The active Response instance
     *
     * @var ResponseInterface
     */
    protected $response;

    /**
     * Handle to the modules Config.
     *
     * @var AppFiltersConfig
     */
    protected $filtersConfig;

    /**
     * Whether we've done initial processing
     * on the filter lists.
     *
     * @var bool
     */
    protected $initialized = false;

    /**
     * The processed filters that will
     * be used to check against.
     *
     * @var array<string, array>
     */
    protected $filters = [
        'before' => [],
        'after'  => [],
    ];

    /**
     * The collection of filters' class names that will
     * be used to execute in each position.
     *
     * @var array<string, array>
     */
    protected $filtersClass = [
        'before' => [],
        'after'  => [],
    ];

    /**
     * Any arguments to be passed to filters.
     *
     * @var array<string, list<string>|null>
     */
    protected $arguments = [];

    /**
     * Constructor.
     *
     * @param FiltersConfig $config
     */
    public function __construct(FiltersConfig $config, RequestInterface $request, ResponseInterface $response)
    {
        $this->config  = $config;
        $this->request = $request;
        $this->setResponse($response);

        if ($config instanceof AppFiltersConfig) {
            $this->filtersConfig = $config;
        } else {
            $this->filtersConfig = config('Filters');
        }
    }

    /**
     * Set the response explicitly.
     */
    public function setResponse(ResponseInterface $response): void
    {
        $this->response = $response;
    }

    /**
     * Runs through all of the filters for the specified
     * uri and position.
     *
     * @param string $uri
     * @param string $position 'before' or 'after'
     *
     * @return RequestInterface|ResponseInterface|string|null
     */
    public function run(string $uri, string $position = 'before')
    {
        $this->initialize(strtolower($uri));

        foreach ($this->filtersClass[$position] as $alias => $rules) {
            $arguments = $this->arguments[$position][$alias] ?? null;

            foreach ($rules as $filter) {
                $instance = $this->getFilterInstance($filter);

                if ($instance === null) {
                    continue;
                }

                if ($position === 'before') {
                    $result = $instance->before($this->request, $arguments);

                    if ($result instanceof RequestInterface) {
                        $this->request = $result;
                    } elseif ($result instanceof ResponseInterface) {
                        return $result;
                    } elseif ($result !== null) {
                        return $result;
                    }
                } elseif ($instance->after($this->request, $this->response, $arguments) instanceof ResponseInterface) {
                    $this->response = $instance->after($this->request, $this->response, $arguments);
                }
            }
        }

        return $position === 'before' ? $this->request : $this->response;
    }

    /**
     * Runs through all of the filters for the specified
     * uri and position.
     *
     * @param string $position 'before' or 'after'
     *
     * @return RequestInterface|ResponseInterface|string|null
     */
    public function runRequired(string $position = 'before')
    {
        $required = $this->filtersConfig->required[$position] ?? [];

        foreach ($required as $filter) {
            $instance = $this->getFilterInstance($filter);

            if ($instance === null) {
                continue;
            }

            if ($position === 'before') {
                $result = $instance->before($this->request, null);

                if ($result instanceof RequestInterface) {
                    $this->request = $result;
                } elseif ($result instanceof ResponseInterface) {
                    return $result;
                } elseif ($result !== null) {
                    return $result;
                }
            } elseif ($instance->after($this->request, $this->response, null) instanceof ResponseInterface) {
                $this->response = $instance->after($this->request, $this->response, null);
            }
        }

        return $position === 'before' ? $this->request : $this->response;
    }

    /**
     * Add a single filter to the chain.
     *
     * @param string|FilterInterface $filter
     * @param string                 $position 'before' or 'after'
     *
     * @return $this
     */
    public function addFilter($filter, string $position = 'before', ?string $alias = null)
    {
        if (! in_array($position, ['before', 'after'], true)) {
            throw new FilterException('Invalid filter position passed: ' . $position);
        }

        if ($filter instanceof FilterInterface) {
            $class = get_class($filter);
        } elseif (is_string($filter)) {
            $class = $this->getFilterClass($filter);
        } else {
            throw new FilterException('Invalid filter type passed.');
        }

        if (! class_exists($class)) {
            throw new FilterException('Filter class not found: ' . $class);
        }

        $alias = $alias ?? strtolower(str_replace('\\', '_', $class));

        if (! isset($this->filtersClass[$position][$alias])) {
            $this->filtersClass[$position][$alias] = [];
        }

        $this->filtersClass[$position][$alias][] = $class;

        return $this;
    }

    /**
     * Enable filters for a specific URI.
     *
     * @param list<string> $filters
     * @param string       $position 'before' or 'after'
     *
     * @return $this
     */
    public function enableFilters(array $filters, string $position = 'before'): self
    {
        if (! in_array($position, ['before', 'after'], true)) {
            throw new FilterException('Invalid filter position passed: ' . $position);
        }

        foreach ($filters as $filter) {
            if (is_string($filter)) {
                $this->addFilter($filter, $position);
            }
        }

        return $this;
    }

    /**
     * Initialize the filters.
     *
     * @param string $uri
     */
    public function initialize(string $uri): self
    {
        if ($this->initialized === true) {
            return $this;
        }

        $this->processGlobals($uri);
        $this->processMethods();
        $this->processFilters($uri);

        $this->initialized = true;

        return $this;
    }

    /**
     * Returns the processed filters array.
     *
     * @return array<string, array>
     */
    public function getFilters(): array
    {
        return $this->filtersClass;
    }

    /**
     * Returns the processed filters class array.
     *
     * @return array<string, array>
     */
    public function getFiltersClass(): array
    {
        $result = [
            'before' => [],
            'after'  => [],
        ];

        foreach ($this->filtersClass['before'] as $alias => $classes) {
            foreach ($classes as $class) {
                $result['before'][] = $class;
            }
        }

        foreach ($this->filtersClass['after'] as $alias => $classes) {
            foreach ($classes as $class) {
                $result['after'][] = $class;
            }
        }

        return $result;
    }

    /**
     * Get required filters for a position.
     *
     * @param string $position 'before' or 'after'
     *
     * @return array{0: list<string>, 1: list<string>}
     */
    public function getRequiredFilters(string $position = 'before'): array
    {
        $required = $this->filtersConfig->required[$position] ?? [];
        $aliases  = [];

        foreach ($required as $filter) {
            if (is_string($filter)) {
                $aliases[] = $filter;
            }
        }

        $classes = [];

        foreach ($aliases as $alias) {
            try {
                $class = $this->getFilterClass($alias);
                $classes[] = $class;
            } catch (FilterException $e) {
                // Skip if filter class not found
            }
        }

        return [$aliases, $classes];
    }

    /**
     * Process the global filters.
     *
     * @param string $uri
     */
    protected function processGlobals(string $uri): void
    {
        if (! isset($this->filtersConfig->globals[$this->request->getMethod()])) {
            $this->filtersConfig->globals[$this->request->getMethod()] = [];
        }

        $globals = $this->filtersConfig->globals[$this->request->getMethod()];

        if ($globals === []) {
            return;
        }

        foreach ($globals as $alias => $rules) {
            $keep = true;

            if (is_array($rules)) {
                $path = $rules['except'] ?? null;

                if ($path !== null) {
                    $keep = ! $this->pathApplies($uri, $path);
                }
            }

            if ($keep) {
                $filter = is_array($rules) ? $alias : $rules;

                if (is_string($filter)) {
                    $this->addFilter($filter, 'before');
                }
            }
        }
    }

    /**
     * Process the method filters.
     */
    protected function processMethods(): void
    {
        if (! isset($this->filtersConfig->methods[$this->request->getMethod()])) {
            return;
        }

        $methods = $this->filtersConfig->methods[$this->request->getMethod()];

        foreach ($methods as $filter) {
            $this->addFilter($filter, 'before');
        }
    }

    /**
     * Process the filters for a specific URI.
     *
     * @param string $uri
     */
    protected function processFilters(string $uri): void
    {
        if ($this->filtersConfig->filters === []) {
            return;
        }

        $uri = strtolower(trim($uri, '/ '));

        foreach ($this->filtersConfig->filters as $alias => $settings) {
            // Look for before rules
            if (isset($settings['before'])) {
                $path = $settings['before'];

                if ($this->pathApplies($uri, $path)) {
                    $this->arguments['before'][$alias] = $settings['arguments'] ?? null;
                    $this->addFilter($alias, 'before');
                }
            }

            // Look for after rules
            if (isset($settings['after'])) {
                $path = $settings['after'];

                if ($this->pathApplies($uri, $path)) {
                    $this->arguments['after'][$alias] = $settings['arguments'] ?? null;
                    $this->addFilter($alias, 'after');
                }
            }
        }
    }

    /**
     * Check if a path applies to a URI.
     *
     * @param string       $uri
     * @param string|array $paths
     */
    protected function pathApplies(string $uri, $paths): bool
    {
        if (empty($uri) || empty($paths)) {
            return false;
        }

        // Make sure the paths are iterable
        if (is_string($paths)) {
            $paths = [$paths];
        }

        // Treat each path as a pseudo-regex
        foreach ($paths as $path) {
            // need to escape path separators
            $path = str_replace('/', '\/', trim($path, '/ '));
            $path = str_replace('*', '.*', $path);

            // Does this rule apply here?
            if (preg_match('#^' . $path . '$#', $uri, $match) === 1) {
                return true;
            }
        }

        return false;
    }

    /**
     * Returns the filter class from an alias.
     *
     * @param string $alias
     *
     * @return string
     */
    protected function getFilterClass(string $alias): string
    {
        if (strpos($alias, '\\') !== false) {
            return $alias;
        }

        if (! isset($this->filtersConfig->aliases[$alias])) {
            throw new FilterException('"' . $alias . '" filter alias is not defined');
        }

        $class = $this->filtersConfig->aliases[$alias];

        if (is_array($class)) {
            $class = $class[0];
        }

        return $class;
    }

    /**
     * Get an instance of a filter.
     *
     * @param string $class
     *
     * @return FilterInterface|null
     */
    protected function getFilterInstance(string $class): ?FilterInterface
    {
        if (! class_exists($class)) {
            return null;
        }

        $instance = new $class();

        if (! $instance instanceof FilterInterface) {
            throw new FilterException('Filter must implement CodeIgniter\Filters\FilterInterface');
        }

        return $instance;
    }
}

