<?php

namespace App\Controllers;

use App\Models\AdminModel;
use CodeIgniter\Database\BaseConnection;

class SetupAdmin extends BaseController
{
    public function index()
    {
        $db = \Config\Database::connect();
        $model = new AdminModel();
        
        // Cek apakah tabel admin sudah ada, jika belum buat tabel
        if (!$db->tableExists('admin')) {
            try {
                $this->createAdminTable($db);
            } catch (\Exception $e) {
                return "Error membuat tabel: " . $e->getMessage() . "<br><br>Pastikan database sudah dikonfigurasi dengan benar di app/Config/Database.php";
            }
        } else {
            // Jika tabel sudah ada, cek apakah kolom created_at dan updated_at ada
            try {
                $fields = $db->getFieldData('admin');
                $fieldNames = array_column($fields, 'name');
                
                if (!in_array('created_at', $fieldNames) || !in_array('updated_at', $fieldNames)) {
                    // Tambahkan kolom yang hilang
                    $this->addMissingColumns($db);
                }
            } catch (\Exception $e) {
                // Jika error, coba buat ulang tabel
                try {
                    $db->query("DROP TABLE IF EXISTS admin");
                    $this->createAdminTable($db);
                } catch (\Exception $e2) {
                    return "Error: " . $e2->getMessage();
                }
            }
        }
        
        // Cek apakah sudah ada admin
        try {
            $admin = $model->first();
            
            if ($admin) {
                return $this->showMessage("Admin sudah ada di database!", "admin", "admin123");
            }
        } catch (\Exception $e) {
            // Jika error, mungkin ada masalah dengan struktur tabel
            return "Error membaca data: " . $e->getMessage() . "<br><br>Coba refresh halaman ini.";
        }
        
        // Buat admin default - hanya gunakan field yang benar-benar ada
        $data = [
            'username' => 'admin',
            'password' => password_hash('admin123', PASSWORD_DEFAULT),
        ];
        
        // Cek apakah kolom timestamp ada sebelum menambahkannya
        try {
            $fields = $db->getFieldData('admin');
            $fieldNames = array_column($fields, 'name');
            
            if (in_array('created_at', $fieldNames)) {
                $data['created_at'] = date('Y-m-d H:i:s');
            }
            if (in_array('updated_at', $fieldNames)) {
                $data['updated_at'] = date('Y-m-d H:i:s');
            }
        } catch (\Exception $e) {
            // Jika tidak bisa cek field, skip timestamp
        }
        
        try {
            $model->insert($data);
            return $this->showMessage("Admin berhasil dibuat!", "admin", "admin123");
        } catch (\Exception $e) {
            return "Error: " . $e->getMessage() . "<br><br>Pastikan database sudah dikonfigurasi dengan benar.";
        }
    }
    
    private function createAdminTable($db)
    {
        $forge = \Config\Database::forge();
        
        $fields = [
            'id' => [
                'type'           => 'INT',
                'constraint'     => 11,
                'unsigned'       => true,
                'auto_increment' => true,
            ],
            'username' => [
                'type'       => 'VARCHAR',
                'constraint' => '100',
            ],
            'password' => [
                'type'       => 'VARCHAR',
                'constraint' => '255',
            ],
            'created_at' => [
                'type' => 'DATETIME',
                'null' => true,
            ],
            'updated_at' => [
                'type' => 'DATETIME',
                'null' => true,
            ],
        ];
        
        $forge->addField($fields);
        $forge->addKey('id', true);
        $forge->createTable('admin', true);
    }
    
    private function addMissingColumns($db)
    {
        $forge = \Config\Database::forge();
        
        try {
            $fields = $db->getFieldData('admin');
            $fieldNames = array_column($fields, 'name');
            
            if (!in_array('created_at', $fieldNames)) {
                $forge->addColumn('admin', [
                    'created_at' => [
                        'type' => 'DATETIME',
                        'null' => true,
                    ],
                ]);
            }
            
            if (!in_array('updated_at', $fieldNames)) {
                $forge->addColumn('admin', [
                    'updated_at' => [
                        'type' => 'DATETIME',
                        'null' => true,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            // Jika tidak bisa menambah kolom, coba dengan query langsung
            try {
                if (!in_array('created_at', $fieldNames)) {
                    $db->query("ALTER TABLE admin ADD COLUMN created_at DATETIME NULL");
                }
                if (!in_array('updated_at', $fieldNames)) {
                    $db->query("ALTER TABLE admin ADD COLUMN updated_at DATETIME NULL");
                }
            } catch (\Exception $e2) {
                throw $e2;
            }
        }
    }
    
    private function showMessage($message, $username, $password)
    {
        return "<!DOCTYPE html>
<html>
<head>
    <title>Setup Admin</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 600px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #28a745;
        }
        .info {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
        }
        .credentials {
            background: #fff3cd;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border-left: 4px solid #ffc107;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
        }
        .btn:hover {
            background: #0056b3;
        }
    </style>
</head>
<body>
    <div class='container'>
        <h1>✓ " . $message . "</h1>
        <div class='info'>
            <p>Gunakan kredensial berikut untuk login:</p>
        </div>
        <div class='credentials'>
            <p><strong>Username:</strong> " . htmlspecialchars($username) . "</p>
            <p><strong>Password:</strong> " . htmlspecialchars($password) . "</p>
        </div>
        <a href='/admin/login' class='btn'>Login Sekarang</a>
    </div>
</body>
</html>";
    }
}

