<?php

namespace App\Controllers;

use App\Models\PegawaiModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\HTTP\Files\UploadedFile;

class PerusahaanAdmin extends BaseController
{
    private PegawaiModel $pegawai;

    public function __construct()
    {
        helper(['text']);
        $this->pegawai = new PegawaiModel();
    }

    public function index(): string
    {
        $search       = trim((string) $this->request->getGet('q'));
        $divisiFilter = (string) $this->request->getGet('divisi');
        $genderFilter = (string) $this->request->getGet('jenis_kelamin');

        $query = $this->pegawai->orderBy('id_pegawai', 'asc');

        if ($search !== '') {
            $query->like('nama_pegawai', $search);
        }

        if ($divisiFilter !== '' && $divisiFilter !== 'semua') {
            $query->where('divisi', $divisiFilter);
        }

        if ($genderFilter !== '' && $genderFilter !== 'semua') {
            $query->where('jenis_kelamin', $genderFilter);
        }

        $pegawai = $query->findAll();

        return view('admin_list_pegawai', [
            'pegawai' => $pegawai,
            'filters' => [
                'search'        => $search,
                'divisi'        => $divisiFilter !== '' ? $divisiFilter : 'semua',
                'jenis_kelamin' => $genderFilter !== '' ? $genderFilter : 'semua',
            ],
        ]);
    }

    public function create()
    {
        if ($this->request->is('post')) {
            $rules = $this->pegawaiRules();

            if (! $this->validate($rules)) {
                return view('admin_create_pegawai', [
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $photoFile = $this->request->getFile('foto_pegawai');
            $photoPath = $this->storePegawaiPhoto($photoFile);

            $payload = [
                'nama_pegawai'  => (string) $this->request->getPost('nama_pegawai'),
                'tanggal_lahir' => (string) $this->request->getPost('tanggal_lahir'),
                'jenis_kelamin' => (string) $this->request->getPost('jenis_kelamin'),
                'divisi'        => (string) $this->request->getPost('divisi'),
                'foto_pegawai'  => $photoPath,
            ];

            $this->pegawai->insert($payload);

            return redirect()
                ->to('/admin/perusahaan')
                ->with('message', 'Data pegawai berhasil ditambahkan.');
        }

        return view('admin_create_pegawai');
    }

    public function edit(int $id)
    {
        $pegawai = $this->findOrFail($id);

        if ($this->request->is('post')) {
            $rules = $this->pegawaiRules();

            if (! $this->validate($rules)) {
                return view('admin_edit_pegawai', [
                    'pegawai'    => $pegawai,
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $removePhoto = (bool) $this->request->getPost('remove_photo');
            $photoFile   = $this->request->getFile('foto_pegawai');
            $newPhoto    = $this->storePegawaiPhoto($photoFile);

            $payload = [
                'nama_pegawai'  => (string) $this->request->getPost('nama_pegawai'),
                'tanggal_lahir' => (string) $this->request->getPost('tanggal_lahir'),
                'jenis_kelamin' => (string) $this->request->getPost('jenis_kelamin'),
                'divisi'        => (string) $this->request->getPost('divisi'),
            ];

            if ($newPhoto !== null) {
                if (! empty($pegawai['foto_pegawai'])) {
                    $this->deletePegawaiPhoto($pegawai['foto_pegawai']);
                }
                $payload['foto_pegawai'] = $newPhoto;
            } elseif ($removePhoto && ! empty($pegawai['foto_pegawai'])) {
                $this->deletePegawaiPhoto($pegawai['foto_pegawai']);
                $payload['foto_pegawai'] = null;
            }

            $this->pegawai->update($id, $payload);

            return redirect()
                ->to('/admin/perusahaan')
                ->with('message', 'Data pegawai berhasil diperbarui.');
        }

        return view('admin_edit_pegawai', [
            'pegawai' => $pegawai,
        ]);
    }

    public function preview(int $id): string
    {
        $pegawai = $this->findOrFail($id);

        $divisiLabels = [
            'Direktur Utama' => 'Direktur Utama',
            'IT'             => 'IT (Information Technology)',
            'Produksi'       => 'Produksi',
            'Marketing'      => 'Marketing',
            'HR'             => 'HR (Human Resource)',
        ];

        $genderLabels = [
            'laki-laki' => 'Laki-laki',
            'perempuan' => 'Perempuan',
        ];

        return view('pegawai_detail', [
            'pegawai'        => $pegawai,
            'divisiLabel'    => $divisiLabels[$pegawai['divisi'] ?? ''] ?? ($pegawai['divisi'] ?? '-'),
            'genderLabel'    => $genderLabels[strtolower($pegawai['jenis_kelamin'] ?? '')] ?? 'Belum diisi',
            'isAdminPreview' => true,
            'showHeader'     => false,
        ]);
    }

    public function photo(int $id): string
    {
        $pegawai = $this->findOrFail($id);

        return view('admin_photo_pegawai', [
            'pegawai' => $pegawai,
        ]);
    }

    public function delete(int $id)
    {
        $pegawai = $this->findOrFail($id);
        $this->pegawai->delete($id);

        if (! empty($pegawai['foto_pegawai'])) {
            $this->deletePegawaiPhoto($pegawai['foto_pegawai']);
        }

        return redirect()
            ->to('/admin/perusahaan')
            ->with('message', 'Data pegawai berhasil dihapus.');
    }

    private function storePegawaiPhoto(?UploadedFile $photoFile): ?string
    {
        if (! $photoFile instanceof UploadedFile) {
            return null;
        }

        if ($photoFile->getError() === UPLOAD_ERR_NO_FILE) {
            return null;
        }

        if (! $photoFile->isValid() || $photoFile->hasMoved()) {
            log_message('error', 'Upload foto pegawai tidak valid: {error}', [
                'error' => $photoFile->getErrorString(),
            ]);

            return null;
        }

        $targetDir = FCPATH . 'uploads/pegawai';
        if (! is_dir($targetDir) && ! mkdir($targetDir, 0755, true) && ! is_dir($targetDir)) {
            log_message('error', 'Gagal membuat direktori foto pegawai: {dir}', ['dir' => $targetDir]);

            return null;
        }

        $newName = $photoFile->getRandomName();

        try {
            $photoFile->move($targetDir, $newName);
        } catch (\Throwable $th) {
            log_message('error', 'Gagal memindahkan foto pegawai: {message}', ['message' => $th->getMessage()]);

            return null;
        }

        return 'uploads/pegawai/' . $newName;
    }

    private function deletePegawaiPhoto(?string $relativePath): void
    {
        if (! $relativePath) {
            return;
        }

        $fullPath = FCPATH . ltrim(str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $relativePath), DIRECTORY_SEPARATOR);
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    private function findOrFail(int $id): array
    {
        $pegawai = $this->pegawai->find($id);

        if (! $pegawai) {
            throw PageNotFoundException::forPageNotFound();
        }

        return $pegawai;
    }

    /**
     * Aturan validasi dan pesan yang digunakan saat membuat/memperbarui pegawai.
     */
    private function pegawaiRules(): array
    {
        return [
            'nama_pegawai'  => [
                'rules'  => 'required|string',
                'errors' => [
                    'required' => 'Nama pegawai wajib diisi.',
                    'string'   => 'Nama pegawai harus berupa teks.',
                ],
            ],
            'tanggal_lahir' => [
                'rules'  => 'required|valid_date',
                'errors' => [
                    'required'   => 'Tanggal lahir wajib diisi.',
                    'valid_date' => 'Tanggal lahir harus menggunakan format tanggal yang benar.',
                ],
            ],
            'jenis_kelamin' => [
                'rules'  => 'required|in_list[laki-laki,perempuan]',
                'errors' => [
                    'required' => 'Jenis kelamin wajib dipilih.',
                    'in_list'  => 'Pilih opsi jenis kelamin yang tersedia.',
                ],
            ],
            'divisi'        => [
                'rules'  => 'required|in_list[Direktur Utama,IT,Produksi,Marketing,HR]',
                'errors' => [
                    'required' => 'Divisi wajib dipilih.',
                    'in_list'  => 'Pilih divisi yang tersedia.',
                ],
            ],
            'foto_pegawai'  => [
                'rules'  => 'permit_empty|is_image[foto_pegawai]|max_size[foto_pegawai,2048]|mime_in[foto_pegawai,image/jpg,image/jpeg,image/png,image/webp]',
                'errors' => [
                    'is_image' => 'Foto pegawai harus berupa gambar.',
                    'max_size' => 'Ukuran foto pegawai maksimal 2 MB.',
                    'mime_in'  => 'Format foto pegawai harus JPG, JPEG, PNG, atau WEBP.',
                ],
            ],
        ];
    }
}
