# Panduan Deployment ke Production

Dokumen ini menjelaskan langkah-langkah untuk mendeploy aplikasi ke server production.

## Prerequisites

- PHP 8.1 atau lebih tinggi
- MySQL/MariaDB
- Web server (Apache/Nginx)
- Composer
- Akses SSH ke server

## Langkah-langkah Deployment

### 1. Persiapan Server

```bash
# Update sistem
sudo apt update && sudo apt upgrade -y

# Install PHP dan ekstensi yang diperlukan
sudo apt install php8.1 php8.1-cli php8.1-fpm php8.1-mysql php8.1-mbstring php8.1-xml php8.1-curl php8.1-zip php8.1-gd

# Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
```

### 2. Upload File ke Server

```bash
# Clone atau upload file ke server
# Pastikan folder berikut ada:
# - app/
# - public/
# - writable/
# - vendor/ (atau install dengan composer)
```

### 3. Konfigurasi Environment

```bash
# Copy file .env.example ke .env (jika ada)
# Atau buat file .env baru dengan konfigurasi berikut:

CI_ENVIRONMENT = production
app.baseURL = 'https://yourdomain.com/'

# Database
database.default.hostname = localhost
database.default.database = your_database
database.default.username = your_username
database.default.password = your_password
database.default.DBDriver = MySQLi

# Generate encryption key
php spark key:generate
```

### 4. Install Dependencies

```bash
# Install Composer dependencies
composer install --no-dev --optimize-autoloader
```

### 5. Konfigurasi Database

```bash
# Jalankan migration
php spark migrate

# (Opsional) Jalankan seeder
php spark db:seed AdminSeeder
```

### 6. Set File Permissions

```bash
# Set permission untuk folder writable
chmod -R 775 writable/
chown -R www-data:www-data writable/

# Set permission untuk folder uploads
chmod -R 755 public/uploads/
chown -R www-data:www-data public/uploads/
```

### 7. Konfigurasi Web Server

#### Apache (.htaccess sudah ada di public/)

Pastikan mod_rewrite diaktifkan:
```bash
sudo a2enmod rewrite
sudo systemctl restart apache2
```

#### Nginx

Tambahkan konfigurasi berikut:

```nginx
server {
    listen 80;
    server_name yourdomain.com;
    root /path/to/your/project/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\. {
        deny all;
    }
}
```

### 8. Konfigurasi SSL (HTTPS)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-apache

# Generate SSL certificate
sudo certbot --apache -d yourdomain.com -d www.yourdomain.com
```

Setelah SSL terpasang, update `.env`:
```
app.forceGlobalSecureRequests = true
```

### 9. Optimasi untuk Production

```bash
# Optimize autoloader
composer dump-autoload --optimize --classmap-authoritative

# Clear cache
php spark cache:clear

# Set environment ke production
# (sudah di-set di .env)
```

### 10. Testing

- [ ] Test login admin
- [ ] Test CRUD berita
- [ ] Test CRUD pegawai
- [ ] Test upload gambar
- [ ] Test halaman public (beranda, berita, about, dll)
- [ ] Test error handling

## Troubleshooting

### Error: Permission Denied
```bash
# Pastikan web server memiliki akses ke folder writable
sudo chown -R www-data:www-data writable/
sudo chmod -R 775 writable/
```

### Error: Database Connection Failed
- Pastikan kredensial database di `.env` benar
- Pastikan database user memiliki privilege yang cukup
- Pastikan MySQL service berjalan

### Error: 500 Internal Server Error
- Cek log error di `writable/logs/`
- Pastikan `CI_ENVIRONMENT = production` di `.env`
- Pastikan file permission sudah benar

### Error: CSRF Token Mismatch
- Pastikan session berfungsi dengan baik
- Pastikan cookie domain di-set dengan benar
- Cek konfigurasi session di `app/Config/Session.php`

## Post-Deployment Checklist

- [ ] Ubah password admin default
- [ ] Backup database dan file
- [ ] Setup automated backup
- [ ] Monitor log error
- [ ] Setup monitoring (opsional)
- [ ] Update dokumentasi dengan URL production

## Maintenance

### Backup Rutin

```bash
# Backup database
mysqldump -u username -p database_name > backup_$(date +%Y%m%d).sql

# Backup file
tar -czf backup_files_$(date +%Y%m%d).tar.gz /path/to/project
```

### Update Aplikasi

```bash
# Pull update dari repository
git pull origin main

# Update dependencies
composer update --no-dev

# Jalankan migration jika ada
php spark migrate

# Clear cache
php spark cache:clear
```




