<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\PendidikanModel;

class PendidikanController extends BaseController
{
    protected $pendidikanModel;

    public function __construct()
    {
        $this->pendidikanModel = new PendidikanModel();
    }

    public function index()
    {
        // Get query parameters
        $search = $this->request->getGet('search');
        $jenjang = $this->request->getGet('jenjang');
        $sort = $this->request->getGet('sort') ?? 'tahun_masuk';
        $order = $this->request->getGet('order') ?? 'DESC';
        $page = $this->request->getGet('page') ?? 1;

        // Build query and paginate
        $pendidikan = $this->pendidikanModel
            ->getPendidikan($search, $jenjang, $sort, $order)
            ->paginate(10, 'default', $page);
        $pager = $this->pendidikanModel->pager;

        $data = [
            'title' => 'Kelola Pendidikan',
            'pendidikan' => $pendidikan,
            'pager' => $pager,
            'search' => $search,
            'jenjang' => $jenjang,
            'sort' => $sort,
            'order' => $order,
            'jenjangList' => $this->pendidikanModel->getJenjangList(),
        ];

        return view('admin/pendidikan/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Pendidikan',
            'jenjangList' => $this->pendidikanModel->getJenjangList(),
        ];

        return view('admin/pendidikan/create', $data);
    }

    public function store()
    {
        $rules = [
            'jenjang'      => 'required|in_list[TK,SD,SMP,SMA,SMK,D1,D2,D3,D4,S1-Sarjana,S2-Magister,S3-Doktor,Profesi,Kursus]',
            'nama_sekolah' => 'required|min_length[3]|max_length[255]',
            'tahun_masuk'  => 'required|numeric|exact_length[4]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'jenjang'      => $this->request->getPost('jenjang'),
            'nama_sekolah' => $this->request->getPost('nama_sekolah'),
            'tahun_masuk'  => $this->request->getPost('tahun_masuk'),
            'tahun_lulus'  => $this->request->getPost('tahun_lulus') ?: null,
            'jurusan'      => $this->request->getPost('jurusan'),
        ];

        $this->pendidikanModel->insert($data);

        return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Data pendidikan berhasil ditambahkan');
    }

    public function edit($id)
    {
        $pendidikan = $this->pendidikanModel->find($id);

        if (!$pendidikan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data pendidikan tidak ditemukan');
        }

        $data = [
            'title' => 'Edit Pendidikan',
            'pendidikan' => $pendidikan,
            'jenjangList' => $this->pendidikanModel->getJenjangList(),
        ];

        return view('admin/pendidikan/edit', $data);
    }

    public function update($id)
    {
        $pendidikan = $this->pendidikanModel->find($id);

        if (!$pendidikan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data pendidikan tidak ditemukan');
        }

        $rules = [
            'jenjang'      => 'required|in_list[TK,SD,SMP,SMA,SMK,D1,D2,D3,D4,S1-Sarjana,S2-Magister,S3-Doktor,Profesi,Kursus]',
            'nama_sekolah' => 'required|min_length[3]|max_length[255]',
            'tahun_masuk'  => 'required|numeric|exact_length[4]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'jenjang'      => $this->request->getPost('jenjang'),
            'nama_sekolah' => $this->request->getPost('nama_sekolah'),
            'tahun_masuk'  => $this->request->getPost('tahun_masuk'),
            'tahun_lulus'  => $this->request->getPost('tahun_lulus') ?: null,
            'jurusan'      => $this->request->getPost('jurusan'),
        ];

        $this->pendidikanModel->update($id, $data);

        return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Data pendidikan berhasil diperbarui');
    }

    public function delete($id)
    {
        $pendidikan = $this->pendidikanModel->find($id);

        if (!$pendidikan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data pendidikan tidak ditemukan');
        }

        $this->pendidikanModel->delete($id);

        return redirect()->to(base_url('admin/pendidikan'))->with('success', 'Data pendidikan berhasil dihapus');
    }
}
