<?php namespace App\Controllers;

use \App\Models\BiodataModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class BiodataAdmin extends BaseController
{
	public function index()
	{
		$model = new BiodataModel();
		
		// Get search parameter
		$search = $this->request->getGet('search');
		
		// Get filter parameters
		$filter_agama = $this->request->getGet('filter_agama');
		
		// Get sorting parameters
		$sort_by = $this->request->getGet('sort_by') ?? 'nama';
		$sort_order = $this->request->getGet('sort_order') ?? 'ASC';
		
		// Apply search
		if (!empty($search)) {
			$model->groupStart()
				->like('nama', $search)
				->orLike('alamat', $search)
				->orLike('email', $search)
				->groupEnd();
		}
		
		// Apply filter
		if (!empty($filter_agama)) {
			$model->where('agama', $filter_agama);
		}
		
		// Apply sorting
		$model->orderBy($sort_by, $sort_order);
		
		$data = [
			'biodata' => $model->paginate(10),
			'pager' => $model->pager,
			'search' => $search,
			'filter_agama' => $filter_agama,
			'sort_by' => $sort_by,
			'sort_order' => $sort_order,
		];
		
		return view('admin/admin_list_biodata', $data);
	}

	//--------------------------------------------------------------------------

	public function preview($id)
	{
		$model = new BiodataModel();
		$data['biodata'] = $model->where('id', $id)->first();
		
		if(!$data['biodata']){
			throw PageNotFoundException::forPageNotFound();
		}
		return view('admin/admin_view_biodata', $data);
	}

	//--------------------------------------------------------------------------

	public function create()
	{
		// lakukan validasi
		$validation = \Config\Services::validation();
		$validation->setRules([
			'nama'          => 'required|min_length[3]|max_length[100]',
			'jenis_kelamin' => 'required',
			'email'         => 'permit_empty|valid_email',
			'tanggal_lahir' => 'required|valid_date'
		]);
		
		$isDataValid = $validation->withRequest($this->request)->run();

		// jika data valid, simpan ke database
		if($isDataValid){
			$model = new BiodataModel();
			
			// Handle file upload for foto_profil
			$foto_profil = '';
			$fileFoto = $this->request->getFile('foto_profil');
			
			// Pastikan folder uploads ada
			if (!is_dir(FCPATH . 'uploads')) {
				mkdir(FCPATH . 'uploads', 0777, true);
			}

			if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
				$newName = $fileFoto->getRandomName();
				$fileFoto->move(FCPATH . 'uploads/', $newName); // Gunakan standard FCPATH public/uploads
				$foto_profil = 'uploads/' . $newName;
			}
			
			$model->insert([
				"nama" => $this->request->getPost('nama'),
				"alamat" => $this->request->getPost('alamat'),
				"tempat_lahir" => $this->request->getPost('tempat_lahir'),
				"tanggal_lahir" => $this->request->getPost('tanggal_lahir'),
				"jenis_kelamin" => $this->request->getPost('jenis_kelamin'),
				"agama" => $this->request->getPost('agama'),
				"no_hp" => $this->request->getPost('no_hp'),
				"email" => $this->request->getPost('email'),
				"foto_profil" => $foto_profil,
				"deskripsi" => $this->request->getPost('deskripsi')
			]);
			
			session()->setFlashdata('message', 'Biodata berhasil ditambahkan');
			return redirect('admin/biodata');
		} else {
		    // Jika tidak valid, kirim error ke view
			if ($this->request->getMethod() === 'post') {
				session()->setFlashdata('errors', $validation->getErrors());
			}
		}
		
		// tampilkan form create
		return view('admin/admin_create_biodata');
	}

	//--------------------------------------------------------------------------

	public function edit($id)
	{
		// ambil data yang akan diedit
		$model = new BiodataModel();
		$data['biodata'] = $model->where('id', $id)->first();
		
		if (!$data['biodata']) {
			throw PageNotFoundException::forPageNotFound();
		}
		
		// lakukan validasi data
		$validation = \Config\Services::validation();
		$validation->setRules([
			'id' => 'required',
			'nama' => 'required|min_length[3]',
			'email' => 'permit_empty|valid_email'
		]);
		
		$isDataValid = $validation->withRequest($this->request)->run();
		
		// jika data valid, maka simpan ke database
		if($isDataValid){
			$updateData = [
				"nama" => $this->request->getPost('nama'),
				"alamat" => $this->request->getPost('alamat'),
				"tempat_lahir" => $this->request->getPost('tempat_lahir'),
				"tanggal_lahir" => $this->request->getPost('tanggal_lahir'),
				"jenis_kelamin" => $this->request->getPost('jenis_kelamin'),
				"agama" => $this->request->getPost('agama'),
				"no_hp" => $this->request->getPost('no_hp'),
				"email" => $this->request->getPost('email'),
				"deskripsi" => $this->request->getPost('deskripsi')
			];
			
			// Handle file upload for foto_profil
			$fileFoto = $this->request->getFile('foto_profil');
			
			// Pastikan folder uploads ada
			if (!is_dir(FCPATH . 'uploads')) {
				mkdir(FCPATH . 'uploads', 0777, true);
			}

			if ($fileFoto && $fileFoto->isValid() && !$fileFoto->hasMoved()) {
				// Delete old foto if exists
				if (!empty($data['biodata']['foto_profil'])) {
					$oldFile = FCPATH . $data['biodata']['foto_profil'];
					if (file_exists($oldFile)) {
						unlink($oldFile);
					}
				}
				$newName = $fileFoto->getRandomName();
				$fileFoto->move(FCPATH . 'uploads/', $newName);
				$updateData['foto_profil'] = 'uploads/' . $newName;
			}
			
			$model->update($id, $updateData);
			session()->setFlashdata('message', 'Biodata berhasil diperbarui');
			return redirect('admin/biodata');
		} else {
			if ($this->request->getMethod() === 'post') {
				session()->setFlashdata('errors', $validation->getErrors());
			}
		}

		// tampilkan form edit
		return view('admin/admin_edit_biodata', $data);
	}

	//--------------------------------------------------------------------------

	public function delete($id){
		$model = new BiodataModel();
		$biodata = $model->where('id', $id)->first();
		
		// Delete file if exist
		if (!empty($biodata['foto_profil'])) {
			$file = WRITEPATH . '../public/' . $biodata['foto_profil'];
			if (file_exists($file)) {
				unlink($file);
			}
		}
		
		$model->delete($id);
		return redirect('admin/biodata');
	}
}

