<?php
/**
 * Test Script untuk RajaOngkir Integration
 * 
 * Cara menjalankan:
 * php test-rajaongkir.php
 */

require __DIR__ . '/vendor/autoload.php';

use Illuminate\Support\Facades\Facade;
use Illuminate\Foundation\Application;

// Bootstrap Laravel
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

echo "========================================\n";
echo "  RAJAONGKIR INTEGRATION TEST\n";
echo "========================================\n\n";

// Test 1: Check if API key is configured
echo "1. Checking RajaOngkir Configuration...\n";
$apiKey = config('services.rajaongkir.api_key');
$accountType = config('services.rajaongkir.account_type', 'starter');
$originCity = config('services.rajaongkir.origin_city_id', '154');

if (empty($apiKey)) {
    echo "   ❌ FAILED: API Key not configured\n";
    echo "   Please add RAJAONGKIR_API_KEY to your .env file\n\n";
} else {
    echo "   ✅ PASSED: API Key configured\n";
    echo "   Account Type: {$accountType}\n";
    echo "   Origin City ID: {$originCity}\n\n";
}

// Test 2: Test RajaOngkirService
echo "2. Testing RajaOngkirService...\n";
try {
    $service = app(App\Services\Shipping\RajaOngkirService::class);
    
    if ($service->isConfigured()) {
        echo "   ✅ PASSED: Service is configured\n\n";
    } else {
        echo "   ❌ FAILED: Service not configured\n\n";
        exit(1);
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n\n";
    exit(1);
}

// Test 3: Get Provinces
echo "3. Testing getProvinces()...\n";
try {
    $provinces = $service->getProvinces();
    
    if (!empty($provinces)) {
        echo "   ✅ PASSED: Retrieved " . count($provinces) . " provinces\n";
        echo "   Sample: " . ($provinces[0]['province'] ?? 'N/A') . "\n\n";
    } else {
        echo "   ❌ FAILED: No provinces retrieved\n";
        echo "   This might indicate API connection issues\n\n";
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n\n";
}

// Test 4: Get Cities
echo "4. Testing getCities()...\n";
try {
    $cities = $service->getCities();
    
    if (!empty($cities)) {
        echo "   ✅ PASSED: Retrieved " . count($cities) . " cities\n";
        $sampleCity = $cities[0] ?? null;
        if ($sampleCity) {
            echo "   Sample: {$sampleCity['type']} {$sampleCity['city_name']}, {$sampleCity['province']}\n\n";
        }
    } else {
        echo "   ❌ FAILED: No cities retrieved\n";
        echo "   This might indicate API connection issues\n\n";
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n\n";
}

// Test 5: Search City
echo "5. Testing searchCity('jakarta')...\n";
try {
    $results = $service->searchCity('jakarta');
    
    if (!empty($results)) {
        echo "   ✅ PASSED: Found " . count($results) . " cities matching 'jakarta'\n";
        foreach (array_slice($results, 0, 3) as $city) {
            echo "   - {$city['type']} {$city['city_name']} (ID: {$city['city_id']})\n";
        }
        echo "\n";
    } else {
        echo "   ❌ FAILED: No cities found for 'jakarta'\n";
        echo "   This is unusual - please check API response\n\n";
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n\n";
}

// Test 6: Search City with different terms
echo "6. Testing searchCity() with various terms...\n";
$testTerms = ['bandung', 'surabaya', 'yogya', 'medan'];

foreach ($testTerms as $term) {
    try {
        $results = $service->searchCity($term);
        $count = count($results);
        
        if ($count > 0) {
            echo "   ✅ '{$term}': Found {$count} result(s)\n";
        } else {
            echo "   ⚠️  '{$term}': No results\n";
        }
    } catch (Exception $e) {
        echo "   ❌ '{$term}': ERROR - " . $e->getMessage() . "\n";
    }
}
echo "\n";

// Test 7: Calculate Shipping Cost
echo "7. Testing getCost() - Jakarta to Bandung...\n";
try {
    // Jakarta Pusat (city_id: 154) to Bandung (city_id: 23)
    $jakartaId = 154;
    $bandungId = 23;
    $weight = 1000; // 1kg
    
    $result = $service->getCost($jakartaId, $bandungId, $weight, 'jne');
    
    if ($result['success'] && !empty($result['data'])) {
        echo "   ✅ PASSED: Retrieved shipping costs\n";
        echo "   Sample cost:\n";
        $sample = $result['data'][0];
        echo "   - {$sample['courier_name']} {$sample['service']}: Rp" . number_format($sample['cost'], 0, ',', '.') . "\n";
        echo "   - ETD: {$sample['etd']} hari\n\n";
    } else {
        echo "   ❌ FAILED: Could not get shipping cost\n";
        echo "   Message: " . ($result['message'] ?? 'Unknown error') . "\n\n";
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n\n";
}

// Summary
echo "========================================\n";
echo "  TEST SUMMARY\n";
echo "========================================\n";

if (empty($apiKey)) {
    echo "\n⚠️  CONFIGURATION REQUIRED:\n";
    echo "   Add this to your .env file:\n\n";
    echo "   RAJAONGKIR_API_KEY=your_api_key_here\n";
    echo "   RAJAONGKIR_ACCOUNT_TYPE=starter\n";
    echo "   RAJAONGKIR_ORIGIN_CITY_ID=154\n\n";
    echo "   Get your API key from: https://rajaongkir.com\n\n";
} else {
    echo "\n✅ All tests completed!\n";
    echo "   If you see any failures above, check:\n";
    echo "   - Internet connection\n";
    echo "   - API key validity\n";
    echo "   - RajaOngkir service status\n";
    echo "   - Laravel logs (storage/logs/laravel.log)\n\n";
}

echo "========================================\n";

