<x-app-layout>
<!-- Set config sebelum x-data - HARUS di body bagian atas -->
<script>
    // Set config langsung di body sebelum x-data
    window.posAppConfig = {
        products: @json($products),
        categories: @json($categories),
        inventory: @json($inventorySummary),
        submitUrl: '{{ route('cashier.orders.store') }}'
    };
    console.log('=== posAppConfig set in body ===');
    console.log('posAppConfig:', window.posAppConfig);
    console.log('Products count:', window.posAppConfig.products?.length || 0);
    
    // Pastikan config ter-set dengan benar
    if (!window.posAppConfig.products || window.posAppConfig.products.length === 0) {
        console.error('ERROR: posAppConfig.products is empty!');
    }
    
    // Inisialisasi posApp data langsung - tanpa menggunakan Alpine.data
    window.posAppData = {
        allProducts: window.posAppConfig.products || [],
        categories: window.posAppConfig.categories || [],
        inventory: window.posAppConfig.inventory || {},
        submitUrl: window.posAppConfig.submitUrl || '',
        fallbackImage: 'https://via.placeholder.com/320x240?text=Menu',
        search: '',
        selectedCategory: 'all',
        filteredProducts: [],
        cart: [],
        customer: { name: 'Pelanggan Umum' },
        notes: '',
        discount: 0,
        discountType: 'amount', // 'amount' or 'percent'
        discountValue: 0,
        orderCode: '',
        isDarkMode: document.documentElement.classList.contains('dark'),
        cashGiven: '',
        quickCashPresets: [10000, 20000, 50000, 100000],
        showPaymentModal: false,
        isProcessing: false,
        
        init() {
            console.log('=== POS App INIT ===');
            console.log('allProducts:', this.allProducts.length);
            
            if (this.allProducts.length > 0) {
                this.filteredProducts = [...this.allProducts];
                console.log('filteredProducts initialized:', this.filteredProducts.length);
            } else {
                console.warn('No products found!');
            }
            
            this.orderCode = this.generateOrderCode();
            this.syncThemeFromStorage();
        },
        
        filterProducts() {
            const keyword = this.search.toLowerCase().trim();
            this.filteredProducts = this.allProducts.filter(product => {
                if (!product) return false;
                
                const matchesKeyword = !keyword || 
                    (product.name && product.name.toLowerCase().includes(keyword)) || 
                    (product.sku && product.sku.toLowerCase().includes(keyword));
                    
                const matchesCategory = this.selectedCategory === 'all' || 
                    product.category_key === this.selectedCategory ||
                    String(product.category_id) === String(this.selectedCategory);
                    
                return matchesKeyword && matchesCategory;
            });
        },
        
        setCategory(categoryId) {
            this.selectedCategory = categoryId;
            this.filterProducts();
        },
        
        generateOrderCode() {
            const now = new Date();
            const pad = (n) => String(n).padStart(2, '0');
            return `POS-${now.getFullYear()}${pad(now.getMonth() + 1)}${pad(now.getDate())}-${pad(now.getHours())}${pad(now.getMinutes())}`;
        },
        
        addItem(product) {
            if (!product || !product.id) return;
            
            const existing = this.cart.find(item => item.id === product.id);
            if (existing) {
                this.increaseQty(existing);
                return;
            }
            
            this.cart.push({
                id: product.id,
                name: product.name || 'Produk',
                sku: product.sku || '',
                price: product.price || 0,
                stock: product.stock ?? 999,
                quantity: 1,
                total: product.price || 0,
            });
        },
        
        removeItem(item) {
            this.cart = this.cart.filter(current => current.id !== item.id);
        },
        
        increaseQty(item) {
            if (item.quantity < item.stock) {
                item.quantity++;
            }
            this.updateManualQty(item);
        },
        
        decreaseQty(item) {
            if (item.quantity > 1) {
                item.quantity--;
                this.updateManualQty(item);
            } else {
                this.removeItem(item);
            }
        },
        
        updateManualQty(item) {
            if (item.quantity < 1 || isNaN(item.quantity)) {
                item.quantity = 1;
            }
            if (item.quantity > item.stock) {
                item.quantity = item.stock;
            }
            item.total = item.quantity * item.price;
        },
        
        get subtotal() {
            return this.cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        },
        
        get discountAmount() {
            if (this.discountType === 'percent') {
                return Math.round(this.subtotal * (this.discountValue / 100));
            }
            return Math.min(this.discountValue, this.subtotal);
        },

        get grandTotal() {
            return Math.max(0, this.subtotal - this.discountAmount);
        },

        applyDiscount() {
            this.discount = this.discountAmount;
        },

        clearDiscount() {
            this.discountValue = 0;
            this.discount = 0;
        },

        get cashReceivedValue() {
            return this.parseAmount(this.cashGiven);
        },

        get changeDue() {
            const diff = this.cashReceivedValue - this.grandTotal;
            return diff > 0 ? diff : 0;
        },

        get paymentGap() {
            const diff = this.grandTotal - this.cashReceivedValue;
            return diff > 0 ? diff : 0;
        },

        get hasShortage() {
            return this.paymentGap > 0;
        },

        get cashStatusMessage() {
            if (!this.cashReceivedValue) {
                return 'Masukkan nominal tunai untuk menghitung kembalian.';
            }
            if (this.hasShortage) {
                return `Uang kurang ${this.formatCurrency(this.paymentGap)}.`;
            }
            if (this.changeDue > 0) {
                return `Kembalian ${this.formatCurrency(this.changeDue)} siap diberikan.`;
            }
            return 'Uang pas, tidak ada kembalian.';
        },

        parseAmount(value) {
            if (typeof value === 'number') return value;
            if (!value) return 0;
            return parseInt(String(value).replace(/[^0-9]/g, ''), 10) || 0;
        },

        applyCashPreset(amount) {
            if (amount === 'exact') {
                this.cashGiven = this.grandTotal > 0 ? this.grandTotal.toLocaleString('id-ID') : '';
                return;
            }
            const current = this.parseAmount(this.cashGiven);
            const total = current + amount;
            this.cashGiven = total > 0 ? total.toLocaleString('id-ID') : '';
        },

        clearCashCalculator() {
            this.cashGiven = '';
        },

        onCashInput(e) {
            const raw = e?.target?.value ?? '';
            const num = this.parseAmount(raw);
            // Keep "0" visible so pengguna tidak mengira input tidak berfungsi
            this.cashGiven = raw.length ? num.toLocaleString('id-ID') : '';
        },

        // Keypad functions
        onKeypadInput(value) {
            if (value === 'CLR') {
                this.clearCashCalculator();
                return;
            }
            
            if (value === 'EXACT') {
                this.applyCashPreset('exact');
                return;
            }
            
            if (value === '-') {
                // Backspace - remove last digit
                const current = String(this.cashGiven).replace(/[^0-9]/g, '');
                if (current.length > 0) {
                    const newValue = current.slice(0, -1);
                    this.cashGiven = newValue.length > 0 ? parseInt(newValue, 10).toLocaleString('id-ID') : '';
                }
                return;
            }
            
            if (value === '00') {
                const current = this.parseAmount(this.cashGiven);
                this.cashGiven = (current * 100).toLocaleString('id-ID');
                return;
            }
            
            if (value === '000') {
                const current = this.parseAmount(this.cashGiven);
                this.cashGiven = (current * 1000).toLocaleString('id-ID');
                return;
            }
            
            // Number input (0-9)
            const current = String(this.cashGiven).replace(/[^0-9]/g, '');
            const newValue = current + String(value);
            // Limit to reasonable amount (999,999,999)
            if (parseInt(newValue, 10) <= 999999999) {
                this.cashGiven = parseInt(newValue, 10).toLocaleString('id-ID');
            }
        },

        openPaymentModal() {
            if (this.cart.length === 0) {
                alert('Keranjang masih kosong. Tambahkan produk terlebih dahulu.');
                return;
            }
            this.showPaymentModal = true;
            // Auto focus on cash input after modal opens
            this.$nextTick(() => {
                const input = document.querySelector('[x-ref="cashInput"]');
                if (input) {
                    setTimeout(() => input.focus(), 100);
                }
            });
        },

        closePaymentModal() {
            this.showPaymentModal = false;
        },

        syncThemeFromStorage() {
            try {
                const savedTheme = localStorage.getItem('theme');
                if (savedTheme) {
                    this.isDarkMode = savedTheme === 'dark';
                }
            } catch (error) {
                console.warn('Failed to sync theme', error);
            }
            document.documentElement.classList.toggle('dark', this.isDarkMode);
        },

        toggleTheme() {
            this.isDarkMode = !this.isDarkMode;
            const theme = this.isDarkMode ? 'dark' : 'light';
            try {
                localStorage.setItem('theme', theme);
            } catch (error) {
                console.warn('Failed to persist theme', error);
            }
            document.documentElement.classList.toggle('dark', this.isDarkMode);
        },
        
        formatCurrency(amount) {
            if (!amount || isNaN(amount)) return 'Rp0';
            return new Intl.NumberFormat('id-ID', { 
                style: 'currency', 
                currency: 'IDR', 
                minimumFractionDigits: 0 
            }).format(amount);
        },
        
        prepareSubmit(event) {
            // Validasi cart
            if (this.cart.length === 0) {
                alert('Keranjang masih kosong. Tambahkan produk terlebih dahulu.');
                return false;
            }
            if (this.hasShortage) {
                alert('Ada produk yang stoknya tidak mencukupi. Periksa kembali keranjang.');
                return false;
            }
            
            // Set processing state
            this.isProcessing = true;
            
            // Cek apakah ini payment modal form
            const isPaymentModalForm = event.target && event.target.id === 'cashier-payment-form';
            
            if (isPaymentModalForm) {
                // Biarkan JavaScript AJAX handler yang handle (sudah prevent default di JS)
                console.log('✅ Payment modal - JavaScript handler will process');
                return; // Tidak submit form biasa
            }
            
            // Form biasa (non-modal), submit normal
            console.log('✅ Regular form - normal submit');
            event.target.submit();
        },

        formatNumber(value) {
            if (!value) return '';
            return String(value).replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        },
    };
    
    console.log('posAppData initialized:', window.posAppData);
</script>

@php
    $lowStockPercent = $inventorySummary['totalStock'] > 0
        ? min(100, round(($inventorySummary['lowStock'] / max(1, $inventorySummary['totalStock'])) * 100))
        : 0;
    $previewProducts = collect($products)->take(3);
@endphp

<div class="cashier-theme min-h-screen bg-gradient-to-br from-[#FFF7F2] via-white to-[#FFEFE6] dark:from-slate-950 dark:via-slate-900 dark:to-slate-900 text-slate-900 dark:text-slate-100"
     x-data="window.posAppData"
     x-init="console.log('Alpine initialized, products:', filteredProducts ? filteredProducts.length : 'N/A')">
    <section class="relative py-5 lg:py-8">
        <div class="mx-auto max-w-6xl space-y-5 px-4 sm:px-6 lg:px-10">
            <div class="relative overflow-hidden rounded-3xl cashier-panel p-5 lg:p-6 border border-[#FAD4C2]/40 dark:border-slate-800 shadow-lg">
                <div class="pointer-events-none absolute inset-0 opacity-20">
                    <div class="absolute -right-10 top-4 h-56 w-56 rounded-full bg-[#F7BFA0]/60 blur-[90px]"></div>
                    <div class="absolute -left-8 bottom-0 h-48 w-48 rounded-full bg-[#FBE0CE]/70 blur-[110px]"></div>
                </div>
                <div class="relative space-y-5">
                    <div class="flex flex-wrap items-center justify-between gap-4">
                        <div>
                            <p class="inline-flex items-center gap-1.5 rounded-lg bg-[#FFE2CF] dark:bg-slate-800 px-2.5 py-1 text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Kasir D'Shop</p>
                            <h1 class="text-2xl lg:text-3xl font-bold text-slate-900 dark:text-white mt-2">POS Kasir</h1>
                            <p class="text-xs text-slate-600 dark:text-slate-300 mt-1">Pilih produk, atur jumlah, lalu simpan transaksi.</p>
                        </div>
                        <div class="flex flex-wrap items-center gap-2.5 text-sm">
                            <button type="button" @click="toggleTheme" class="flex items-center gap-1.5 rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-3 py-1.5 text-xs font-semibold text-[#C26A3D] dark:text-amber-200 shadow-sm transition hover:bg-[#FFE2CF] dark:hover:bg-slate-700">
                                <svg x-show="!isDarkMode" class="h-3.5 w-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24" x-cloak><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12.79A9 9 0 1111.21 3 7 7 0 0021 12.79z" /></svg>
                                <svg x-show="isDarkMode" class="h-3.5 w-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24" x-cloak><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 3v2m0 14v2m9-9h-2M5 12H3m15.364 6.364-1.414-1.414M7.05 7.05 5.636 5.636m12.728 0-1.414 1.414M7.05 16.95l-1.414 1.414M12 8a4 4 0 100 8 4 4 0 000-8z" /></svg>
                                <span x-show="!isDarkMode" x-cloak>Gelap</span>
                                <span x-show="isDarkMode" x-cloak>Terang</span>
                            </button>
                            @auth
                                <div class="flex items-center gap-2.5 rounded-xl border border-[#FAD4C2]/60 dark:border-slate-700 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-3 py-2 shadow-sm">
                                    <div class="flex h-8 w-8 items-center justify-center rounded-lg bg-gradient-to-br from-[#C26A3D] to-[#F5703B] text-white text-xs font-bold shadow-sm">
                                        {{ \Illuminate\Support\Str::take(Auth::user()->name, 1) }}
                                    </div>
                                    <div class="flex flex-col min-w-0">
                                        <span class="text-[10px] font-semibold text-[#C26A3D] dark:text-amber-200 truncate">Kasir</span>
                                        <span class="text-xs font-bold text-slate-900 dark:text-white truncate">{{ Auth::user()->name }}</span>
                                    </div>
                                    <div class="flex items-center gap-1.5 ml-1">
                                        <a href="{{ route('profile.edit') }}" class="rounded-lg border border-[#F7BFA0] bg-white dark:bg-slate-800 px-2 py-1 text-[10px] font-medium text-[#C26A3D] dark:text-amber-200 transition hover:bg-[#FFF4EC] dark:hover:bg-slate-700">Profil</a>
                                        <form method="POST" action="{{ route('logout') }}">
                                            @csrf
                                            <button type="submit" class="rounded-lg border border-rose-200 dark:border-rose-400/60 bg-rose-50 dark:bg-rose-500/20 px-2 py-1 text-[10px] font-medium text-rose-500 dark:text-rose-200 transition hover:bg-rose-100 dark:hover:bg-rose-500/30">Keluar</button>
                                        </form>
                                    </div>
                                </div>
                            @endauth
                            <div class="rounded-xl border border-[#F7BFA0]/60 dark:border-slate-700 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-3 py-2 shadow-sm">
                                <p class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Hari Ini</p>
                                <p class="text-base font-bold text-emerald-600 dark:text-emerald-300">Rp{{ number_format($todayIncome, 0, ',', '.') }}</p>
                            </div>
                            <div class="rounded-xl border border-[#F7BFA0]/60 dark:border-slate-700 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-3 py-2 shadow-sm">
                                <p class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Pesanan</p>
                                <p class="text-base font-bold text-[#C26A3D] dark:text-amber-200">{{ $todayOrders->count() }}</p>
                            </div>
                        </div>
                    </div>
                    <div class="grid gap-3 text-sm sm:grid-cols-3">
                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-4 py-3 shadow-sm">
                            <p class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Varian Produk</p>
                            <p class="mt-1.5 text-xl font-bold text-slate-900 dark:text-white">{{ number_format($inventorySummary['totalProducts']) }}</p>
                        </div>
                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-4 py-3 shadow-sm">
                            <p class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Total Stok</p>
                            <p class="mt-1.5 text-xl font-bold text-emerald-600 dark:text-emerald-300">{{ number_format($inventorySummary['totalStock']) }}</p>
                        </div>
                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm px-4 py-3 shadow-sm">
                            <p class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200">Stok Menipis</p>
                            <p class="mt-1.5 text-xl font-bold text-amber-500 dark:text-amber-300">{{ number_format($inventorySummary['lowStock']) }}</p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid gap-6 xl:grid-cols-[2.4fr,1.1fr]">
                <div class="space-y-4">
                    <div class="rounded-2xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm p-4 shadow-lg space-y-3">
                        <div class="space-y-2">
                            <div class="relative">
                                <input x-model="search" @input="filterProducts" type="search" placeholder="Cari produk..." class="w-full rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-10 py-2.5 text-sm text-slate-900 placeholder:text-slate-400 focus:border-[#C26A3D] focus:ring-2 focus:ring-[#C26A3D]/20 dark:text-white transition" />
                                <svg class="pointer-events-none absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-[#C26A3D] dark:text-amber-200" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="m21 21-4.35-4.35m0 0a7.5 7.5 0 1 0-10.607-10.607 7.5 7.5 0 0 0 10.607 10.607Z" /></svg>
                            </div>
                            <select x-model="selectedCategory" @change="filterProducts" class="w-full rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-3 py-2.5 text-sm text-slate-900 focus:border-[#C26A3D] focus:ring-2 focus:ring-[#C26A3D]/20 dark:text-white transition">
                                <option value="all">Semua Kategori</option>
                                <template x-for="category in categories" :key="`option-${category.id}`">
                                    <option :value="category.id" x-text="category.name"></option>
                                </template>
                            </select>
                        </div>
                        <div class="flex flex-wrap gap-1.5 overflow-x-auto pb-1">
                            <button type="button" @click="setCategory('all')" class="rounded-lg border px-3 py-1.5 text-xs font-semibold transition whitespace-nowrap" :class="selectedCategory === 'all' ? 'border-[#C26A3D] bg-[#FFE2CF] text-[#C26A3D] dark:border-amber-300 dark:bg-slate-800 dark:text-amber-200' : 'border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 text-slate-600 dark:text-slate-200 hover:bg-[#FFF4EC] dark:hover:bg-slate-700'">
                                Semua
                            </button>
                            <template x-for="category in categories" :key="'chip-' + category.id">
                                <button type="button" @click="setCategory(category.id)" class="rounded-lg border px-3 py-1.5 text-xs font-semibold transition whitespace-nowrap" :class="selectedCategory === category.id ? 'border-[#C26A3D] bg-[#FFE2CF] text-[#C26A3D] dark:border-amber-300 dark:bg-slate-800 dark:text-amber-200' : 'border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 text-slate-600 dark:text-slate-200 hover:bg-[#FFF4EC] dark:hover:bg-slate-700'">
                                    <span x-text="category.name"></span>
                                </button>
                            </template>
                        </div>

                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm p-3 max-h-[500px] overflow-y-auto shadow-inner">
                            <template x-if="filteredProducts.length === 0">
                                <p class="py-8 text-center text-sm text-slate-500 dark:text-slate-300">Produk tidak ditemukan.</p>
                            </template>
                            <div class="grid gap-3 md:grid-cols-2 xl:grid-cols-3">
                                <template x-for="product in filteredProducts" :key="product.id">
                                    <button type="button" @click="addItem(product)" class="group flex min-h-[200px] flex-col gap-2 rounded-xl border bg-white p-3 text-left shadow-md transition duration-200 hover:-translate-y-0.5 hover:shadow-lg dark:bg-slate-900/70" :class="product.stock <= 5 ? 'border-amber-300 dark:border-amber-600' : 'border-[#FAD4C2]/60 dark:border-slate-700'">
                                        <div class="relative aspect-[4/3] overflow-hidden rounded-lg bg-[#FFF0E3] dark:bg-slate-800">
                                            <img :src="product.thumbnail || fallbackImage" :alt="product.name" class="h-full w-full object-cover transition duration-300 group-hover:scale-105" loading="lazy" />
                                            <span class="absolute left-2 top-2 rounded-lg bg-white/95 dark:bg-slate-900/95 backdrop-blur-sm px-2 py-0.5 text-[9px] font-bold text-[#C26A3D] dark:text-amber-200 shadow-sm" x-text="product.category_name || 'Umum'"></span>
                                        </div>
                                        <div class="space-y-0.5 flex-1">
                                            <p class="text-sm font-bold text-slate-900 line-clamp-2 dark:text-white" x-text="product.name"></p>
                                            <p class="text-[10px] text-slate-500 dark:text-slate-400" x-text="product.sku || 'SKU: -'"></p>
                                        </div>
                                        <div class="flex items-center justify-between pt-1">
                                            <p class="text-base font-bold text-[#C26A3D] dark:text-amber-200" x-text="formatCurrency(product.price)"></p>
                                            <span class="rounded-lg border px-2 py-0.5 text-[10px] font-semibold" :class="product.stock <= 5 ? 'border-amber-300 bg-amber-50 text-amber-600 dark:border-amber-600 dark:bg-amber-900/30 dark:text-amber-300' : 'border-[#FAD4C2]/60 dark:border-slate-600 bg-slate-50 dark:bg-slate-800 text-slate-600 dark:text-slate-300'" x-text="product.stock"></span>
                                        </div>
                                    </button>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="space-y-4">
                    <form method="POST" :action="submitUrl" @submit.prevent="prepareSubmit" class="space-y-3">
                        @csrf
                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm p-4 shadow-lg space-y-3">
                            <div class="grid gap-2 sm:grid-cols-2">
                                <div>
                                    <label class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200 mb-1 block">No. Bon</label>
                                    <input type="text" x-model="orderCode" name="order_code" class="w-full rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-3 py-2 text-sm text-slate-900 focus:border-[#C26A3D] focus:ring-2 focus:ring-[#C26A3D]/20 dark:text-white transition" />
                                </div>
                                <div>
                                    <label class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200 mb-1 block">Status</label>
                                    <input type="hidden" name="payment_status" value="paid">
                                    <div class="rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-slate-50 dark:bg-slate-800 px-3 py-2 text-sm text-slate-900 dark:text-white">
                                        Sudah Dibayar
                                    </div>
                                </div>
                            </div>
                            <div>
                                <label class="text-[10px] font-bold uppercase tracking-wider text-[#C26A3D] dark:text-amber-200 mb-1 block">Nama Pelanggan</label>
                                <input type="text" x-model="customer.name" name="customer_name" placeholder="Pelanggan Umum" class="w-full rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-3 py-2 text-sm text-slate-900 placeholder:text-slate-400 focus:border-[#C26A3D] focus:ring-2 focus:ring-[#C26A3D]/20 dark:text-white transition" />
                            </div>
                        </div>

                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm p-4 shadow-lg space-y-3">
                            <div class="flex items-center justify-between">
                                <p class="text-sm font-bold text-slate-900 dark:text-white">Keranjang</p>
                                <span class="rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-[#FFE2CF] dark:bg-slate-800 px-2.5 py-1 text-[10px] font-bold text-[#C26A3D] dark:text-amber-200" x-text="cart.length + ' item'"></span>
                            </div>
                            <div class="max-h-64 overflow-y-auto rounded-lg border border-[#FAD4C2]/40 dark:border-slate-800 bg-slate-50/50 dark:bg-slate-900/50">
                                <template x-if="cart.length === 0">
                                    <p class="px-4 py-8 text-center text-sm text-slate-500 dark:text-slate-300">Keranjang kosong.</p>
                                </template>
                                <div class="divide-y divide-[#FAD4C2]/40 dark:divide-slate-800">
                                    <template x-for="(item, index) in cart" :key="item.id">
                                        <div class="p-3 space-y-2">
                                            <div class="flex items-start justify-between gap-2">
                                                <div class="flex-1 min-w-0">
                                                    <p class="font-bold text-sm text-slate-900 dark:text-white truncate" x-text="item.name"></p>
                                                    <p class="text-[10px] text-slate-500 dark:text-slate-400" x-text="item.sku || 'SKU: -'"></p>
                                                </div>
                                                <button type="button" class="rounded-lg border border-rose-200 dark:border-rose-500/40 bg-rose-50 dark:bg-rose-500/20 px-2 py-1 text-[10px] font-bold text-rose-500 dark:text-rose-200 transition hover:bg-rose-100 dark:hover:bg-rose-500/30" @click="removeItem(item)">Hapus</button>
                                            </div>
                                            <div class="flex items-center justify-between gap-2">
                                                <div class="flex items-center gap-1.5 rounded-lg border border-[#FAD4C2]/60 dark:border-slate-700 bg-white dark:bg-slate-800 px-2 py-1">
                                                    <button type="button" @click="decreaseQty(item)" class="px-1.5 text-sm leading-none text-[#C26A3D] dark:text-amber-200 font-bold">-</button>
                                                    <input type="number" min="1" :max="item.stock" x-model.number="item.quantity" @input="updateManualQty(item)" class="w-10 border-none bg-transparent text-center text-xs font-bold text-slate-900 no-spinner dark:text-white" />
                                                    <button type="button" @click="increaseQty(item)" class="px-1.5 text-sm leading-none text-[#C26A3D] dark:text-amber-200 font-bold">+</button>
                                                </div>
                                                <div class="text-right">
                                                    <p class="text-xs text-slate-500 dark:text-slate-400" x-text="formatCurrency(item.price)"></p>
                                                    <p class="text-sm font-bold text-emerald-600 dark:text-emerald-300" x-text="formatCurrency(item.total)"></p>
                                                </div>
                                            </div>
                                            <input type="hidden" :name="`items[${index}][product_id]`" :value="item.id">
                                            <input type="hidden" :name="`items[${index}][quantity]`" :value="item.quantity">
                                        </div>
                                    </template>
                                </div>
                            </div>
                        </div>

                        <div class="rounded-xl border border-[#FAD4C2]/40 dark:border-slate-800 bg-white/90 dark:bg-slate-900/90 backdrop-blur-sm p-4 shadow-lg space-y-3">
                            <div class="space-y-2 text-sm">
                                <div class="flex items-center justify-between">
                                    <span class="text-xs font-semibold text-slate-600 dark:text-slate-400">Subtotal</span>
                                    <span class="text-sm font-bold text-slate-900 dark:text-white" x-text="formatCurrency(subtotal)"></span>
                                </div>
                                <div class="flex items-center justify-between gap-2">
                                    <span class="text-xs font-semibold text-slate-600 dark:text-slate-400 flex-1">Diskon</span>
                                    <div class="flex items-center gap-1.5 flex-1">
                                        <select x-model="discountType" @change="clearDiscount()" class="text-[10px] border border-[#FAD4C2]/60 dark:border-slate-700 rounded-lg px-2 py-1 bg-white dark:bg-slate-800 text-slate-700 dark:text-slate-200">
                                            <option value="amount">Rp</option>
                                            <option value="percent">%</option>
                                        </select>
                                        <input type="number" 
                                               x-model.number="discountValue" 
                                               @input="applyDiscount()"
                                               min="0"
                                               :max="discountType === 'percent' ? 100 : subtotal"
                                               placeholder="0"
                                               class="w-16 text-xs border border-[#FAD4C2]/60 dark:border-slate-700 rounded-lg px-2 py-1 bg-white dark:bg-slate-800 text-right text-slate-900 dark:text-white" />
                                        <button type="button" @click="clearDiscount()" class="text-rose-500 hover:text-rose-600 transition" x-show="discountValue > 0" x-cloak>
                                            <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                        </button>
                                    </div>
                                    <span class="text-sm font-bold text-emerald-600 dark:text-emerald-300 text-right" x-text="formatCurrency(discountAmount)"></span>
                                </div>
                                <div class="flex items-center justify-between pt-2 border-t border-[#FAD4C2]/40 dark:border-slate-700">
                                    <span class="text-sm font-bold text-[#C26A3D] dark:text-amber-200">Total Bayar</span>
                                    <span class="text-lg font-bold text-[#C26A3D] dark:text-amber-200" x-text="formatCurrency(grandTotal)"></span>
                                </div>
                            </div>

                            <div class="space-y-3">
                                <div class="grid gap-2 sm:grid-cols-2">
                                    <div class="rounded-lg border border-[#FAD4C2]/40 dark:border-slate-800 bg-slate-50/50 dark:bg-slate-900/50 p-3">
                                        <p class="text-[10px] font-bold uppercase tracking-wider text-slate-600 dark:text-slate-400 mb-1">Total Belanja</p>
                                        <p class="text-lg font-bold text-slate-900 dark:text-white" x-text="formatCurrency(grandTotal)"></p>
                                    </div>
                                    <div class="rounded-lg border border-[#FAD4C2]/40 dark:border-slate-800 bg-slate-50/50 dark:bg-slate-900/50 p-3">
                                        <p class="text-[10px] font-bold uppercase tracking-wider text-slate-600 dark:text-slate-400 mb-1">Tunai Diterima</p>
                                        <p class="text-lg font-bold text-[#C26A3D] dark:text-amber-200" x-text="formatCurrency(cashReceivedValue)"></p>
                                    </div>
                                </div>
                                
                                <div class="flex items-center justify-between rounded-lg border-2 p-3" :class="hasShortage ? 'border-rose-300 bg-rose-50 dark:border-rose-500 dark:bg-rose-900/20' : 'border-emerald-300 bg-emerald-50 dark:border-emerald-500 dark:bg-emerald-900/20'">
                                    <div>
                                        <p class="text-[10px] font-bold uppercase tracking-wider" :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : 'text-emerald-600 dark:text-emerald-300'" x-text="hasShortage ? 'Kurang' : 'Kembalian'"></p>
                                        <p class="text-base font-bold" :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : 'text-emerald-600 dark:text-emerald-300'" x-text="hasShortage ? formatCurrency(paymentGap) : formatCurrency(changeDue)"></p>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-[10px] font-bold uppercase tracking-wider text-slate-600 dark:text-slate-300">Dibayar</p>
                                        <p class="text-base font-bold text-[#C26A3D] dark:text-amber-200" x-text="formatCurrency(cashReceivedValue)"></p>
                                    </div>
                                </div>

                                <div class="space-y-2 pt-1">
                                    <button type="button" @click="openPaymentModal()" class="w-full rounded-lg bg-gradient-to-r from-[#C26A3D] to-[#F5703B] px-4 py-3 text-sm font-bold text-white shadow-md transition hover:shadow-lg hover:from-[#a85327] hover:to-[#C26A3D]" :class="{'opacity-50 cursor-not-allowed': cart.length === 0}" :disabled="cart.length === 0">
                                        Buka Pembayaran
                                    </button>
                                    <input type="hidden" name="print" value="1">
                                    <input type="hidden" name="cash_received" :value="cashReceivedValue">
                                    <input type="hidden" name="cash_change" :value="changeDue">
                                    <input type="hidden" name="discount_amount" :value="discountAmount">
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </section>

    <!-- Payment Modal Overlay -->
    <div x-show="showPaymentModal" 
         x-cloak
         @keydown.escape.window="closePaymentModal()"
         class="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm p-2 sm:p-4 overflow-y-auto"
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition ease-in duration-200"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0">
        <div @click.away="closePaymentModal()" 
             class="relative w-full max-w-lg my-auto rounded-2xl bg-white shadow-2xl dark:bg-slate-900 max-h-[95vh] flex flex-col"
             x-transition:enter="transition ease-out duration-300"
             x-transition:enter-start="opacity-0 scale-95"
             x-transition:enter-end="opacity-100 scale-100"
             x-transition:leave="transition ease-in duration-200"
             x-transition:leave-start="opacity-100 scale-100"
             x-transition:leave-end="opacity-0 scale-95">
            <div class="p-4 space-y-3 overflow-y-auto flex-1 payment-modal-content">
                <!-- Header -->
                <div class="flex items-center justify-between border-b border-slate-200 dark:border-slate-700 pb-2">
                    <h2 class="text-lg font-bold text-slate-900 dark:text-white">Pembayaran Tunai</h2>
                    <button type="button" @click="closePaymentModal()" class="rounded-full p-1.5 text-slate-500 hover:bg-slate-100 dark:hover:bg-slate-800 dark:text-slate-400">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>

                <!-- Payment Summary -->
                <div class="grid gap-2 grid-cols-2">
                    <div class="rounded-lg border border-slate-200 bg-slate-50 p-2 dark:border-slate-700 dark:bg-slate-800">
                        <p class="text-[9px] font-semibold uppercase tracking-[0.15em] text-slate-500 dark:text-slate-400 mb-0.5">Cash Received</p>
                        <p class="text-base font-bold text-slate-900 dark:text-white" x-text="formatCurrency(cashReceivedValue)"></p>
                    </div>
                    <div class="rounded-lg border border-slate-200 bg-slate-50 p-2 dark:border-slate-700 dark:bg-slate-800">
                        <p class="text-[9px] font-semibold uppercase tracking-[0.15em] text-slate-500 dark:text-slate-400 mb-0.5">Dibayar</p>
                        <p class="text-base font-bold text-[#C26A3D] dark:text-amber-200" x-text="formatCurrency(cashReceivedValue)"></p>
                    </div>
                </div>

                <!-- Summary Details -->
                <div class="rounded-lg border border-slate-200 bg-slate-50 p-2 space-y-1 dark:border-slate-700 dark:bg-slate-800">
                    <div class="flex items-center justify-between text-xs">
                        <span class="text-slate-600 dark:text-slate-400">Subtotal</span>
                        <span class="font-semibold text-slate-900 dark:text-white" x-text="formatCurrency(subtotal)"></span>
                    </div>
                    <div class="flex items-center justify-between text-xs" x-show="discountAmount > 0">
                        <span class="text-slate-600 dark:text-slate-400">Diskon</span>
                        <span class="font-semibold text-emerald-600 dark:text-emerald-300" x-text="'-' + formatCurrency(discountAmount)"></span>
                    </div>
                    <div class="flex items-center justify-between text-xs pt-1 border-t border-slate-200 dark:border-slate-700">
                        <span class="font-semibold text-slate-700 dark:text-slate-300">Total Belanja</span>
                        <span class="font-bold text-lg text-slate-900 dark:text-white" x-text="formatCurrency(grandTotal)"></span>
                    </div>
                </div>

                <!-- Total & Change Display -->
                <div class="rounded-lg border-2 p-2" :class="hasShortage ? 'border-rose-300 bg-rose-50 dark:border-rose-500 dark:bg-rose-900/20' : 'border-emerald-300 bg-emerald-50 dark:border-emerald-500 dark:bg-emerald-900/20'">
                    <div class="flex items-center justify-between">
                        <p class="text-[10px] font-semibold" :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : 'text-emerald-600 dark:text-emerald-300'" x-text="hasShortage ? 'Kurang Dibayar' : 'Kembalian'"></p>
                        <p class="text-lg font-bold" :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : 'text-emerald-600 dark:text-emerald-300'" x-text="hasShortage ? formatCurrency(paymentGap) : formatCurrency(changeDue)"></p>
                    </div>
                </div>

                <!-- Cash Input Display -->
                <div class="rounded-lg border-2 p-2 transition-colors" 
                     :class="hasShortage ? 'border-rose-400 bg-rose-50 dark:border-rose-500 dark:bg-rose-900/20' : cashReceivedValue >= grandTotal ? 'border-emerald-400 bg-emerald-50 dark:border-emerald-500 dark:bg-emerald-900/20' : 'border-[#C26A3D] bg-white dark:border-amber-400 dark:bg-slate-800'">
                    <p class="text-[9px] font-semibold uppercase tracking-[0.15em] mb-1" 
                       :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : cashReceivedValue >= grandTotal ? 'text-emerald-600 dark:text-emerald-300' : 'text-[#C26A3D] dark:text-amber-200'">CASH RECEIVED</p>
                    <div class="flex items-center gap-1.5">
                        <span class="text-base font-bold" 
                              :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : cashReceivedValue >= grandTotal ? 'text-emerald-600 dark:text-emerald-300' : 'text-[#C26A3D] dark:text-amber-200'">Rp</span>
                        <input type="text" 
                               x-ref="cashInput"
                               x-model="cashGiven" 
                               @input="onCashInput($event)" 
                               @keydown.escape="closePaymentModal()"
                               @keydown.enter.prevent="!hasShortage && cart.length > 0 && document.querySelector('form[action=\'' + submitUrl + '\']')?.requestSubmit()"
                               inputmode="numeric" 
                               placeholder="0" 
                               autocomplete="off" 
                               class="flex-1 text-lg font-bold bg-transparent border-none outline-none"
                               :class="hasShortage ? 'text-rose-600 dark:text-rose-300' : cashReceivedValue >= grandTotal ? 'text-emerald-600 dark:text-emerald-300' : 'text-slate-900 dark:text-white'"
                               readonly />
                        <button type="button" 
                                @click="clearCashCalculator()" 
                                class="rounded-full p-1 text-rose-500 hover:bg-rose-50 dark:hover:bg-rose-900/20 transition"
                                x-show="cashGiven"
                                x-cloak>
                            <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Numeric Keypad -->
                <div class="grid grid-cols-4 gap-2">
                    <!-- Row 1: 7, 8, 9, CLR -->
                    <button type="button" @click="onKeypadInput(7)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">7</button>
                    <button type="button" @click="onKeypadInput(8)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">8</button>
                    <button type="button" @click="onKeypadInput(9)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">9</button>
                    <button type="button" @click="onKeypadInput('CLR')" class="flex items-center justify-center rounded-lg border border-rose-200 bg-rose-50 py-3 text-sm font-bold text-rose-600 shadow-sm transition hover:bg-rose-100 active:scale-95 dark:border-rose-900/50 dark:bg-rose-900/20 dark:text-rose-400 dark:hover:bg-rose-900/30">CLR</button>
                    
                    <!-- Row 2: 4, 5, 6, - -->
                    <button type="button" @click="onKeypadInput(4)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">4</button>
                    <button type="button" @click="onKeypadInput(5)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">5</button>
                    <button type="button" @click="onKeypadInput(6)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">6</button>
                    <button type="button" @click="onKeypadInput('-')" class="flex items-center justify-center rounded-lg border border-slate-200 bg-slate-50 py-3 text-lg font-bold text-slate-600 shadow-sm transition hover:bg-slate-100 active:scale-95 dark:border-slate-700 dark:bg-slate-800/50 dark:text-slate-300 dark:hover:bg-slate-700">⌫</button>
                    
                    <!-- Row 3: 1, 2, 3, 000 -->
                    <button type="button" @click="onKeypadInput(1)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">1</button>
                    <button type="button" @click="onKeypadInput(2)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">2</button>
                    <button type="button" @click="onKeypadInput(3)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">3</button>
                    <button type="button" @click="onKeypadInput('000')" class="flex items-center justify-center rounded-lg border border-slate-200 bg-slate-50 py-3 text-sm font-bold text-slate-600 shadow-sm transition hover:bg-slate-100 active:scale-95 dark:border-slate-700 dark:bg-slate-800/50 dark:text-slate-300 dark:hover:bg-slate-700">000</button>
                    
                    <!-- Row 4: 0, 00, EXACT -->
                    <button type="button" @click="onKeypadInput(0)" class="flex items-center justify-center rounded-lg border border-slate-200 bg-white py-3 text-lg font-bold text-slate-700 shadow-sm transition hover:bg-slate-50 active:scale-95 dark:border-slate-700 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">0</button>
                    <button type="button" @click="onKeypadInput('00')" class="flex items-center justify-center rounded-lg border border-slate-200 bg-slate-50 py-3 text-sm font-bold text-slate-600 shadow-sm transition hover:bg-slate-100 active:scale-95 dark:border-slate-700 dark:bg-slate-800/50 dark:text-slate-300 dark:hover:bg-slate-700">00</button>
                    <button type="button" @click="onKeypadInput('EXACT')" class="col-span-2 flex items-center justify-center rounded-lg bg-[#C26A3D] py-3 text-sm font-bold text-white shadow-sm transition hover:bg-[#a85327] active:scale-95 dark:bg-amber-600 dark:hover:bg-amber-700">UANG PAS</button>
                </div>

                <!-- Quick Cash Buttons -->
                <div class="grid grid-cols-4 gap-1.5">
                    <template x-for="preset in quickCashPresets" :key="`quick-${preset}`">
                        <button type="button" 
                                @click="applyCashPreset(preset)" 
                                class="quick-cash-btn"
                                x-text="formatCurrency(preset)"></button>
                    </template>
                </div>

            </div>
            
            <!-- Action Buttons - Sticky Bottom -->
            <div class="p-4 pt-0 space-y-2 border-t border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900 rounded-b-2xl">
                <form method="POST" :action="submitUrl" id="cashier-payment-form">
                    @csrf
                    <input type="hidden" name="print" value="1">
                    <input type="hidden" name="cash_received" :value="cashReceivedValue">
                    <input type="hidden" name="cash_change" :value="changeDue">
                    <input type="hidden" name="payment_status" value="paid">
                    <input type="hidden" name="customer_name" :value="customer.name">
                    <input type="hidden" name="notes" :value="notes">
                    <input type="hidden" name="order_code" :value="orderCode">
                    <input type="hidden" name="discount_amount" :value="discountAmount">
                    <template x-for="(item, index) in cart" :key="item.id">
                        <input type="hidden" :name="`items[${index}][product_id]`" :value="item.id">
                        <input type="hidden" :name="`items[${index}][quantity]`" :value="item.quantity">
                    </template>
                    <button type="submit" 
                            class="w-full rounded-lg bg-gradient-to-r from-emerald-500 to-emerald-600 px-3 py-2.5 text-sm font-bold text-white shadow-md transition hover:from-emerald-600 hover:to-emerald-700 hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
                            :disabled="cart.length === 0 || hasShortage || isProcessing">
                        <svg x-show="!isProcessing" class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                        </svg>
                        <svg x-show="isProcessing" class="animate-spin w-4 h-4" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        <span x-text="isProcessing ? 'Memproses...' : 'Complete Payment'"></span>
                    </button>
                </form>
                <button type="button" 
                        @click="closePaymentModal()"
                        class="w-full rounded-lg border-2 border-slate-300 bg-white px-3 py-2 text-sm font-semibold text-slate-700 transition hover:bg-slate-50 dark:border-slate-600 dark:bg-slate-800 dark:text-slate-200 dark:hover:bg-slate-700">
                    Batal
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Handler pembayaran kasir: rebuild items dari cart Alpine, kirim via AJAX, redirect ke struk
document.addEventListener('DOMContentLoaded', () => {
    const paymentForm = document.getElementById('cashier-payment-form');
    if (!paymentForm) return;

    paymentForm.addEventListener('submit', async (event) => {
        event.preventDefault();

        if (paymentForm.classList.contains('is-fetching')) return false;

        const alpineRoot = paymentForm.closest('[x-data]');
        const alpineData = alpineRoot ? Alpine.$data(alpineRoot) : null;
        const cartItems = alpineData && Array.isArray(alpineData.cart) ? alpineData.cart : [];

        if (!cartItems.length) {
            alert('Keranjang masih kosong!');
            return false;
        }
        if (alpineData && alpineData.hasShortage) {
            alert('Ada produk yang stoknya tidak mencukupi!');
            return false;
        }

        if (alpineData) alpineData.isProcessing = true;
        paymentForm.classList.add('is-fetching');

        const formData = new FormData(paymentForm);
        Array.from(formData.keys())
            .filter(key => key.startsWith('items['))
            .forEach(key => formData.delete(key));

        cartItems.forEach((item, index) => {
            formData.append(`items[${index}][product_id]`, item.id);
            formData.append(`items[${index}][quantity]`, item.quantity);
        });

        formData.set('payment_status', 'paid');
        formData.set('print', '1');

        try {
            const response = await fetch(paymentForm.action, {
                method: 'POST',
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: formData
            });

            const data = await response.json();

            if (!response.ok || data.status === 'error') {
                const detail = data.errors ? Object.values(data.errors).flat().join('\n') : '';
                throw new Error((data.message || 'Gagal menyimpan transaksi') + (detail ? '\n' + detail : ''));
            }

            if (data.print_url) {
                window.location.href = data.print_url;
                return true;
            }

            if (data.order_id) {
                window.location.href = `/kasir/orders/${data.order_id}/print?auto_print=1`;
                return true;
            }

            window.location.reload();
            return true;
        } catch (error) {
            alert('Gagal menyimpan transaksi:\n' + error.message);
            paymentForm.classList.remove('is-fetching');
            if (alpineData) alpineData.isProcessing = false;
            return false;
        }
    });
});
</script>
</x-app-layout>
