import './bootstrap';

import Alpine from 'alpinejs';

window.Alpine = Alpine;

// Jangan start Alpine jika sudah di-load dari CDN (untuk cashier view)
// Untuk cashier view, Alpine.js sudah di-start di layout
if (window.location.pathname.includes('/kasir')) {
    console.log('Cashier view detected - Alpine.js will be started by CDN script');
} else {
    // Start Alpine dari Vite (untuk non-cashier views)
    Alpine.start();
}

document.addEventListener('DOMContentLoaded', () => {
    // Theme toggle
    const applyTheme = (mode) => {
        const isDark = mode === 'dark';
        document.documentElement.classList.toggle('dark', isDark);
        try { localStorage.setItem('theme', isDark ? 'dark' : 'light'); } catch (e) {}
    };
    const savedTheme = (() => { try { return localStorage.getItem('theme'); } catch (e) { return null; } })();
    if (savedTheme === 'dark' || savedTheme === 'light') {
        applyTheme(savedTheme);
    }
    document.querySelectorAll('[data-theme-toggle]').forEach((btn) => {
        btn.addEventListener('click', () => {
            const isDark = document.documentElement.classList.contains('dark');
            applyTheme(isDark ? 'light' : 'dark');
        });
    });

    const csrfToken =
        document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') ?? '';

    const cartBadges = document.querySelectorAll('[data-cart-count]');
    const cartLinks = document.querySelectorAll('[data-cart-link]');

    const updateCartCount = (count) => {
        cartBadges.forEach((badge) => {
            if (count > 0) {
                badge.textContent = count > 99 ? '99+' : String(count);
                badge.classList.remove('hidden');
            } else {
                badge.classList.add('hidden');
            }
        });
    };

    const bounceCart = () => {
        cartLinks.forEach((link) => {
            link.classList.add('cart-bounce');
            setTimeout(() => link.classList.remove('cart-bounce'), 600);
        });
    };

    const createToastContainer = () => {
        let container = document.getElementById('cart-toast-stack');
        if (!container) {
            container = document.createElement('div');
            container.id = 'cart-toast-stack';
            container.className =
                'fixed top-4 right-4 z-[9999] flex flex-col gap-3 max-w-xs sm:max-w-sm';
            document.body.appendChild(container);
        }

        return container;
    };

    const showToast = (message) => {
        const container = createToastContainer();
        const toast = document.createElement('div');
        toast.className =
            'cart-toast translate-x-0 opacity-100 rounded-2xl border border-[#FAD4C2] bg-white px-4 py-3 text-sm font-semibold text-slate-700 shadow-lg ring-1 ring-black/5 transition';
        toast.innerHTML = `
            <div class="flex items-start gap-3">
                <div class="mt-0.5 flex h-8 w-8 items-center justify-center rounded-full bg-[#FFE2CF] text-[#C26A3D]">
                    <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                    </svg>
                </div>
                <p class="leading-tight">${message}</p>
            </div>
        `;

        container.appendChild(toast);

        setTimeout(() => {
            toast.classList.add('toast-leave');
            setTimeout(() => toast.remove(), 300);
        }, 2600);
    };

    const animateToCart = (imageUrl, trigger) => {
        const cartTarget = cartLinks[0];
        if (!cartTarget || !trigger) {
            return;
        }

        const startRect = trigger.getBoundingClientRect();
        const endRect = cartTarget.getBoundingClientRect();

        const element = document.createElement(imageUrl ? 'img' : 'div');

        if (imageUrl && element instanceof HTMLImageElement) {
            element.src = imageUrl;
        } else {
            element.className =
                'bg-[#C26A3D] text-white flex items-center justify-center text-xs font-semibold';
            element.textContent = 'Keranjang';
        }

        element.style.position = 'fixed';
        element.style.pointerEvents = 'none';
        element.style.zIndex = '9999';
        element.style.left = `${startRect.left + startRect.width / 2 - 40}px`;
        element.style.top = `${startRect.top + startRect.height / 2 - 40}px`;
        element.style.width = '80px';
        element.style.height = '80px';
        element.style.borderRadius = '9999px';
        element.style.boxShadow = '0 12px 30px rgba(0,0,0,0.18)';
        element.style.overflow = 'hidden';
        element.style.transform = 'translate(0, 0) scale(1)';
        document.body.appendChild(element);

        const deltaX = endRect.left + endRect.width / 2 - (startRect.left + startRect.width / 2);
        const deltaY = endRect.top + endRect.height / 2 - (startRect.top + startRect.height / 2);

        element.animate(
            [
                { transform: 'translate(0, 0) scale(1)', opacity: 1 },
                { transform: `translate(${deltaX}px, ${deltaY}px) scale(0.35)`, opacity: 0.2 },
            ],
            {
                duration: 700,
                easing: 'cubic-bezier(0.22, 1, 0.36, 1)',
            },
        ).onfinish = () => element.remove();
    };

    document.querySelectorAll('[data-product-actions]').forEach((section) => {
        const quantityInput = section.querySelector('[data-quantity-input]');
        const cartQuantity = section.querySelector('[data-cart-quantity]');
        const buyNowQuantity = section.querySelector('[data-buy-now-quantity]');

        const syncQuantity = () => {
            if (!quantityInput) {
                return;
            }

            const min = quantityInput.min ? parseInt(quantityInput.min, 10) : 1;
            let value = parseInt(quantityInput.value, 10);
            if (Number.isNaN(value)) {
                value = min;
            }
            value = Math.max(min, value);

            if (quantityInput.max) {
                value = Math.min(parseInt(quantityInput.max, 10), value);
            }

            quantityInput.value = String(value);

            if (cartQuantity) {
                cartQuantity.value = String(value);
            }
            if (buyNowQuantity) {
                buyNowQuantity.value = String(value);
            }
        };

        if (quantityInput) {
            quantityInput.addEventListener('input', syncQuantity);
            quantityInput.addEventListener('change', syncQuantity);
            syncQuantity();
        }
    });

    document.querySelectorAll('[data-add-to-cart-form]').forEach((form) => {
        const submitButton = form.querySelector('[data-add-to-cart-button]');
        const productName = form.dataset.productName ?? 'Produk';
        const productImage = form.dataset.productImage ?? null;

        form.addEventListener('submit', async (event) => {
            event.preventDefault();

            if (!submitButton) {
                form.submit();
                return;
            }

            const formData = new FormData(form);

            try {
                submitButton.disabled = true;
                submitButton.classList.add('opacity-70', 'cursor-not-allowed');

                const response = await fetch(form.action, {
                    method: 'POST',
                    headers: {
                        Accept: 'application/json',
                        'X-Requested-With': 'XMLHttpRequest',
                        ...(csrfToken ? { 'X-CSRF-TOKEN': csrfToken } : {}),
                    },
                    body: formData,
                });

                if (!response.ok) {
                    if (response.status === 419 || response.status === 401) {
                        window.location.reload();
                        return;
                    }
                    throw new Error('Failed to add to cart');
                }

        const data = await response.json();
        const parsedCount = Number(data.cart_count);
        const cartCount = Number.isFinite(parsedCount) ? parsedCount : 0;
        updateCartCount(cartCount);
        showToast(data.message ?? `${productName} berhasil ditambahkan ke keranjang.`);
                animateToCart(productImage, submitButton);
                bounceCart();
            } catch (error) {
                console.error(error);
                showToast('Gagal menambahkan ke keranjang. Coba lagi.');
            } finally {
                submitButton.disabled = false;
                submitButton.classList.remove('opacity-70', 'cursor-not-allowed');
            }
        });
    });

    const gallery = document.querySelector('[data-product-gallery]');
    if (gallery) {
        const mainImage = gallery.querySelector('[data-gallery-main]');
        const thumbnails = gallery.querySelectorAll('[data-gallery-thumb]');
        const openButton = gallery.querySelector('[data-gallery-open]');

        const swapImage = (url) => {
            if (!mainImage) {
                return;
            }
            mainImage.src = url;
        };

        thumbnails.forEach((thumb) => {
            thumb.addEventListener('click', () => {
                const image = thumb.dataset.image;
                if (image) {
                    swapImage(image);
                }
            });
        });

        const createLightbox = () => {
            let overlay = document.getElementById('product-lightbox');
            if (overlay) {
                return overlay;
            }

            overlay = document.createElement('div');
            overlay.id = 'product-lightbox';
            overlay.className =
                'fixed inset-0 z-[9998] hidden items-center justify-center bg-black/70 backdrop-blur-sm p-4';
            overlay.innerHTML = `
                <div class="relative max-h-full w-full max-w-3xl">
                    <button type="button" class="absolute -top-12 right-0 text-white hover:text-[#FFE2CF]" data-lightbox-close>
                        <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                    <img src="" alt="Detail produk" class="mx-auto max-h-[80vh] w-auto rounded-3xl border border-[#FAD4C2] bg-white object-contain p-4 shadow-2xl" data-lightbox-image>
                </div>
            `;

            document.body.appendChild(overlay);

            overlay.addEventListener('click', (event) => {
                if (event.target === overlay) {
                    overlay.classList.add('hidden');
                }
            });

            const closeBtn = overlay.querySelector('[data-lightbox-close]');
            closeBtn?.addEventListener('click', () => overlay?.classList.add('hidden'));

            return overlay;
        };

        const openLightbox = () => {
            const overlay = createLightbox();
            if (!overlay) {
                return;
            }

            const image = overlay.querySelector('[data-lightbox-image]');
            if (image && mainImage) {
                image.src = mainImage.src;
                overlay.classList.remove('hidden');
            }
        };

        openButton?.addEventListener('click', openLightbox);
    }
});

(() => {
    const PRELOADER_ID = 'dshop-preloader';
    const PRELOADER_SHOWN_KEY =
        window.__dshopPreloaderLocalStorageKey ?? 'dshop.preloader-shown';
    let preloaderHidden = false;

    const hidePreloader = () => {
        if (preloaderHidden) {
            return;
        }

        const preloader = document.getElementById(PRELOADER_ID);
        if (preloader) {
            try {
                localStorage.setItem(PRELOADER_SHOWN_KEY, '1');
            } catch (error) {}
        }
        if (!preloader) {
            preloaderHidden = true;
            return;
        }

        preloaderHidden = true;
        preloader.classList.add('dshop-preloader--hidden');
        setTimeout(() => {
            preloader.remove();
        }, 650);
    };

    if (document.readyState === 'complete') {
        hidePreloader();
    } else {
        window.addEventListener('load', hidePreloader);
    }

    setTimeout(hidePreloader, 2600);
})();
