<?php

namespace App\Services\Payment;

use App\Models\Order;
use App\Models\Payment;
use Illuminate\Support\Str;
use Midtrans\Config;
use Midtrans\Snap;

class PaymentGatewayService
{
    public function __construct()
    {
        // Set Midtrans configuration
        Config::$serverKey = config('services.midtrans.server_key');
        Config::$isProduction = config('services.midtrans.is_production', false);
        Config::$isSanitized = config('services.midtrans.is_sanitized', true);
        Config::$is3ds = config('services.midtrans.is_3ds', true);
    }

    public function createSnapToken(Order $order): Payment
    {
        // Check if Midtrans is configured
        if (empty(config('services.midtrans.server_key'))) {
            // Fallback to simulated payment if Midtrans not configured
            return $this->createSimulatedPayment($order);
        }

        try {
            // Prepare transaction details
            $params = [
                'transaction_details' => [
                    'order_id' => $order->invoice_number,
                    'gross_amount' => $order->grand_total,
                ],
                'customer_details' => [
                    'first_name' => $order->customer_name ?? 'Pelanggan',
                    'email' => $order->user?->email ?? data_get($order->shipping_address, 'phone'),
                    'phone' => data_get($order->shipping_address, 'phone'),
                    'billing_address' => [
                        'first_name' => $order->customer_name ?? 'Pelanggan',
                        'address' => data_get($order->shipping_address, 'address_line1', ''),
                        'city' => data_get($order->shipping_address, 'city', ''),
                        'postal_code' => data_get($order->shipping_address, 'postal_code', ''),
                        'country_code' => 'IDN',
                    ],
                    'shipping_address' => [
                        'first_name' => data_get($order->shipping_address, 'recipient_name', $order->customer_name ?? 'Pelanggan'),
                        'address' => data_get($order->shipping_address, 'address_line1', ''),
                        'city' => data_get($order->shipping_address, 'city', ''),
                        'postal_code' => data_get($order->shipping_address, 'postal_code', ''),
                        'country_code' => 'IDN',
                    ],
                ],
                'item_details' => $order->items->map(function ($item) {
                    return [
                        'id' => $item->product_id,
                        'price' => $item->unit_price,
                        'quantity' => $item->quantity,
                        'name' => $item->product_name,
                    ];
                })->toArray(),
                'expiry' => [
                    'start_time' => now()->format('Y-m-d H:i:s O'),
                    'unit' => 'day',
                    'duration' => 1,
                ],
            ];

            // Add shipping cost as item if exists
            if ($order->shipping_cost > 0) {
                $params['item_details'][] = [
                    'id' => 'SHIPPING',
                    'price' => $order->shipping_cost,
                    'quantity' => 1,
                    'name' => 'Biaya Pengiriman (' . strtoupper($order->shipping_courier ?? 'JNE') . ')',
                ];
            }

            // Add discount as item if exists
            if ($order->discount_amount > 0) {
                $params['item_details'][] = [
                    'id' => 'DISCOUNT',
                    'price' => -$order->discount_amount,
                    'quantity' => 1,
                    'name' => 'Diskon',
                ];
            }

            // Create Snap transaction
            $snapToken = Snap::getSnapToken($params);

            // Midtrans Snap API returns UUID format tokens, which is correct
            // No need to validate UUID format - Midtrans tokens are UUIDs
            
            // Create payment record
            $payment = $order->payments()->create([
                'amount' => $order->grand_total,
                'method' => 'midtrans',
                'gateway' => 'midtrans',
                'status' => 'waiting_confirmation',
                'reference' => $order->invoice_number,
                'snap_token' => $snapToken,
                'payment_url' => (config('services.midtrans.is_production') ? 'https://app.midtrans.com' : 'https://app.sandbox.midtrans.com') . '/snap/v4/redirection/' . $snapToken,
                'payload' => $params,
            ]);

            return $payment;
        } catch (\Exception $e) {
            // Log error with full details
            \Log::error('Midtrans Snap Token creation failed', [
                'error' => $e->getMessage(),
                'code' => $e->getCode(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'order_id' => $order->invoice_number,
                'server_key_set' => !empty(config('services.midtrans.server_key')),
                'server_key_preview' => substr(config('services.midtrans.server_key', ''), 0, 20) . '...',
            ]);
            
            return $this->createSimulatedPayment($order);
        }
    }

    protected function createSimulatedPayment(Order $order): Payment
    {
        $payment = $order->payments()->create([
            'amount' => $order->grand_total,
            'method' => 'midtrans',
            'gateway' => 'midtrans',
            'status' => 'waiting_confirmation',
            'reference' => 'SNAP' . strtoupper(Str::random(10)),
            'payment_url' => 'https://app.midtrans.com/snap/v3/redirection/' . Str::random(16),
            'snap_token' => Str::uuid()->toString(),
            'payload' => [
                'message' => 'Simulated Midtrans transaction. Set MIDTRANS_SERVER_KEY and MIDTRANS_CLIENT_KEY in .env to use real payments.',
            ],
        ]);

        return $payment;
    }

    public function markAsPaid(Payment $payment): Payment
    {
        $payment->update([
            'status' => 'paid',
            'paid_at' => now(),
        ]);

        $payment->order()->update([
            'payment_status' => 'paid',
            'status' => $payment->order->status === 'pending' ? 'confirmed' : $payment->order->status,
            'paid_at' => now(),
        ]);

        $order = $payment->order->fresh();

        // Send email notification
        if ($order->user && $order->user->email) {
            try {
                \Illuminate\Support\Facades\Mail::to($order->user->email)
                    ->send(new \App\Mail\PaymentSuccessNotification($order));
            } catch (\Exception $e) {
                \Log::error('Failed to send payment success email: ' . $e->getMessage(), [
                    'order_id' => $order->id,
                    'email' => $order->user->email,
                ]);
            }
        }

        return $payment->fresh();
    }

    public function handleWebhook(array $payload): ?Payment
    {
        $orderId = $payload['order_id'] ?? null;
        $transactionStatus = $payload['transaction_status'] ?? null;
        $fraudStatus = $payload['fraud_status'] ?? null;

        if (!$orderId) {
            return null;
        }

        // Find payment by invoice number
        $payment = Payment::where('reference', $orderId)
            ->orWhereHas('order', function ($query) use ($orderId) {
                $query->where('invoice_number', $orderId);
            })
            ->first();

        if (!$payment) {
            return null;
        }

        // Handle different transaction statuses
        switch ($transactionStatus) {
            case 'capture':
                if ($fraudStatus === 'accept') {
                    $this->markAsPaid($payment);
                }
                break;

            case 'settlement':
                $this->markAsPaid($payment);
                break;

            case 'pending':
                $payment->update([
                    'status' => 'waiting_confirmation',
                ]);
                break;

            case 'deny':
            case 'expire':
            case 'cancel':
                $payment->update([
                    'status' => 'failed',
                ]);
                $payment->order()->update([
                    'payment_status' => 'failed',
                ]);
                break;
        }

        return $payment->fresh();
    }
}

