<?php

namespace App\Services\Notification;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected string $accessToken;
    protected string $phoneNumberId;
    protected string $baseUrl = 'https://graph.facebook.com/v18.0';

    public function __construct()
    {
        $this->accessToken = config('services.whatsapp.access_token', '');
        $this->phoneNumberId = config('services.whatsapp.phone_number_id', '');
    }

    /**
     * Send text message via WhatsApp Business API
     */
    public function sendMessage(string $to, string $message): array
    {
        try {
            if (!$this->isConfigured()) {
                Log::warning('WhatsApp not configured');
                return [
                    'success' => false,
                    'message' => 'WhatsApp API belum dikonfigurasi',
                ];
            }

            // Format phone number (remove + and any spaces)
            $to = preg_replace('/[^0-9]/', '', $to);
            
            // Add country code if not present (assuming Indonesia)
            if (!str_starts_with($to, '62')) {
                if (str_starts_with($to, '0')) {
                    $to = '62' . substr($to, 1);
                } else {
                    $to = '62' . $to;
                }
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Content-Type' => 'application/json',
            ])->post("{$this->baseUrl}/{$this->phoneNumberId}/messages", [
                'messaging_product' => 'whatsapp',
                'recipient_type' => 'individual',
                'to' => $to,
                'type' => 'text',
                'text' => [
                    'preview_url' => false,
                    'body' => $message,
                ],
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                Log::info('WhatsApp message sent successfully', [
                    'to' => $to,
                    'message_id' => $data['messages'][0]['id'] ?? null,
                ]);

                return [
                    'success' => true,
                    'message' => 'Pesan berhasil dikirim',
                    'data' => $data,
                ];
            }

            Log::error('WhatsApp send message error', [
                'to' => $to,
                'status' => $response->status(),
                'body' => $response->json(),
            ]);

            return [
                'success' => false,
                'message' => 'Gagal mengirim pesan WhatsApp',
                'error' => $response->json(),
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp send message exception: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send template message
     */
    public function sendTemplateMessage(string $to, string $templateName, array $parameters = []): array
    {
        try {
            if (!$this->isConfigured()) {
                return [
                    'success' => false,
                    'message' => 'WhatsApp API belum dikonfigurasi',
                ];
            }

            // Format phone number
            $to = preg_replace('/[^0-9]/', '', $to);
            if (!str_starts_with($to, '62')) {
                if (str_starts_with($to, '0')) {
                    $to = '62' . substr($to, 1);
                } else {
                    $to = '62' . $to;
                }
            }

            $components = [];
            if (!empty($parameters)) {
                $components[] = [
                    'type' => 'body',
                    'parameters' => array_map(function ($param) {
                        return [
                            'type' => 'text',
                            'text' => $param,
                        ];
                    }, $parameters),
                ];
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Content-Type' => 'application/json',
            ])->post("{$this->baseUrl}/{$this->phoneNumberId}/messages", [
                'messaging_product' => 'whatsapp',
                'to' => $to,
                'type' => 'template',
                'template' => [
                    'name' => $templateName,
                    'language' => [
                        'code' => 'id',
                    ],
                    'components' => $components,
                ],
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'message' => 'Template berhasil dikirim',
                    'data' => $response->json(),
                ];
            }

            Log::error('WhatsApp send template error', [
                'to' => $to,
                'template' => $templateName,
                'status' => $response->status(),
                'body' => $response->json(),
            ]);

            return [
                'success' => false,
                'message' => 'Gagal mengirim template WhatsApp',
                'error' => $response->json(),
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp send template exception: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send payment success notification
     */
    public function sendPaymentSuccessNotification($order): array
    {
        $customerPhone = $order->shipping_address['phone'] ?? $order->user->phone ?? null;

        if (!$customerPhone) {
            return [
                'success' => false,
                'message' => 'Nomor telepon customer tidak tersedia',
            ];
        }

        $message = $this->formatPaymentSuccessMessage($order);

        return $this->sendMessage($customerPhone, $message);
    }

    /**
     * Format payment success message
     */
    protected function formatPaymentSuccessMessage($order): string
    {
        $storeName = config('app.name', 'DShop');
        $invoiceNumber = $order->invoice_number;
        $total = number_format($order->grand_total, 0, ',', '.');
        $customerName = $order->customer_name ?? 'Customer';

        $message = "🎉 *PEMBAYARAN BERHASIL* 🎉\n\n";
        $message .= "Halo *{$customerName}*,\n\n";
        $message .= "Terima kasih! Pembayaran Anda telah kami terima.\n\n";
        $message .= "📋 *Detail Pesanan:*\n";
        $message .= "• No. Invoice: {$invoiceNumber}\n";
        $message .= "• Total Pembayaran: Rp {$total}\n";
        $message .= "• Status: Pembayaran Berhasil ✅\n\n";
        
        $message .= "📦 Pesanan Anda sedang diproses dan akan segera dikirim.\n\n";
        
        $message .= "Untuk melihat detail pesanan, silakan login ke akun Anda.\n\n";
        $message .= "Terima kasih telah berbelanja di {$storeName}! 🛍️";

        return $message;
    }

    /**
     * Send order shipped notification
     */
    public function sendOrderShippedNotification($order, $trackingNumber = null): array
    {
        $customerPhone = $order->shipping_address['phone'] ?? $order->user->phone ?? null;

        if (!$customerPhone) {
            return [
                'success' => false,
                'message' => 'Nomor telepon customer tidak tersedia',
            ];
        }

        $storeName = config('app.name', 'DShop');
        $invoiceNumber = $order->invoice_number;
        $customerName = $order->customer_name ?? 'Customer';
        $courier = $order->shipping_courier ?? '-';
        $service = $order->shipping_service ?? '-';

        $message = "📦 *PESANAN DIKIRIM* 📦\n\n";
        $message .= "Halo *{$customerName}*,\n\n";
        $message .= "Pesanan Anda telah dikirim!\n\n";
        $message .= "📋 *Detail Pengiriman:*\n";
        $message .= "• No. Invoice: {$invoiceNumber}\n";
        $message .= "• Kurir: {$courier} ({$service})\n";
        
        if ($trackingNumber) {
            $message .= "• No. Resi: {$trackingNumber}\n";
        }
        
        $message .= "\nPesanan Anda sedang dalam perjalanan. Mohon ditunggu ya! 🚚\n\n";
        $message .= "Terima kasih telah berbelanja di {$storeName}! 🛍️";

        return $this->sendMessage($customerPhone, $message);
    }

    /**
     * Check if WhatsApp is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->accessToken) && !empty($this->phoneNumberId);
    }
}








