<?php

namespace App\Services\Cart;

use App\Models\Cart;
use App\Models\CartItem;
use Illuminate\Http\Request;

class CartService
{
    public function resolve(Request $request, bool $createIfMissing = false): ?Cart
    {
        $user = $request->user();
        $sessionId = $request->session()->getId();

        $cart = Cart::query()
            ->when($user, fn ($query) => $query->where('user_id', $user->id))
            ->when(! $user, fn ($query) => $query->whereNull('user_id')->where('session_id', $sessionId))
            ->first();

        if (! $cart) {
            $cart = Cart::query()
                ->where('session_id', $sessionId)
                ->when($user, fn ($query) => $query->orWhere('user_id', $user->id))
                ->first();
        }

        if (! $cart && $createIfMissing) {
            $cart = Cart::create([
                'user_id' => $user?->id,
                'session_id' => $sessionId,
                'status' => 'active',
                'expires_at' => now()->addWeek(),
                'last_activity_at' => now(),
            ]);
        } elseif ($cart && $user && $cart->user_id !== $user->id) {
            $cart->update(['user_id' => $user->id]);
        }

        return $cart;
    }

    public function selectedItems(Cart $cart)
    {
        return $cart->items()
            ->with('product.category')
            ->where('is_selected', true)
            ->get();
    }

    public function clearSelectedItems(Cart $cart): void
    {
        $cart->items()->where('is_selected', true)->delete();

        if ($cart->items()->count() === 0) {
            $cart->update(['status' => 'checked_out']);
        } else {
            $cart->refreshTotals();
        }
    }
}

