<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Order extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $fillable = [
        'invoice_number',
        'user_id',
        'cashier_id',
        'cart_id',
        'channel',
        'status',
        'payment_status',
        'payment_method',
        'payment_gateway',
        'payment_reference',
        'payment_charge_party',
        'total_items',
        'subtotal_amount',
        'discount_amount',
        'shipping_cost',
        'service_fee',
        'grand_total',
        'shipping_address',
        'shipping_courier',
        'expedition_preference',
        'shipping_service',
        'tracking_number',
        'notes',
        'payment_due_at',
        'paid_at',
        'fulfilled_at',
        'metadata',
    ];

    protected $casts = [
        'shipping_address' => 'array',
        'metadata' => 'array',
        'payment_due_at' => 'datetime',
        'paid_at' => 'datetime',
        'fulfilled_at' => 'datetime',
    ];

    protected static function booted(): void
    {
        static::creating(function (Order $order) {
            if (empty($order->invoice_number)) {
                $order->invoice_number = self::generateInvoiceNumber();
            }
        });
    }

    public static function generateInvoiceNumber(): string
    {
        $prefix = 'DS-' . now()->format('Ymd');
        $random = strtoupper(Str::random(5));

        return "{$prefix}-{$random}";
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function cashier()
    {
        return $this->belongsTo(User::class, 'cashier_id');
    }

    public function cart()
    {
        return $this->belongsTo(Cart::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function scopeStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function getCustomerNameAttribute(): ?string
    {
        if (array_key_exists('customer_name', $this->metadata ?? [])) {
            return $this->metadata['customer_name'];
        }

        if (array_key_exists('recipient_name', $this->shipping_address ?? [])) {
            return $this->shipping_address['recipient_name'];
        }

        return $this->user?->name;
    }

    public function getNetIncomeAttribute(): int
    {
        return (int) max($this->grand_total - $this->shipping_cost, 0);
    }
}
