<?php

namespace App\Http\Controllers;

use App\Services\Payment\PaymentGatewayService;
use App\Services\Notification\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MidtransWebhookController extends Controller
{
    public function __construct(
        protected PaymentGatewayService $paymentGateway,
        protected WhatsAppService $whatsappService
    ) {
    }

    /**
     * Handle Midtrans webhook callback
     */
    public function handle(Request $request)
    {
        try {
            $payload = $request->all();

            // Log webhook payload for debugging
            Log::info('Midtrans Webhook received', [
                'payload' => $payload,
                'order_id' => $payload['order_id'] ?? null,
                'transaction_status' => $payload['transaction_status'] ?? null,
            ]);

            // Verify webhook signature (optional but recommended)
            // You can implement signature verification here if needed

            // Handle webhook
            $payment = $this->paymentGateway->handleWebhook($payload);

            if ($payment) {
                // Send notifications if payment is successful
                if (in_array($payment->status, ['paid', 'success']) && $payment->order) {
                    $order = $payment->order->fresh();
                    
                    // Send WhatsApp notification
                    try {
                        $this->whatsappService->sendPaymentSuccessNotification($order);
                        Log::info('WhatsApp payment notification sent', [
                            'order_id' => $order->id,
                            'invoice' => $order->invoice_number,
                        ]);
                    } catch (\Exception $e) {
                        // Log error but don't fail webhook
                        Log::error('Failed to send WhatsApp notification: ' . $e->getMessage(), [
                            'order_id' => $order->id,
                        ]);
                    }
                    
                    // Send email notification
                    if ($order->user && $order->user->email) {
                        try {
                            \Illuminate\Support\Facades\Mail::to($order->user->email)
                                ->send(new \App\Mail\PaymentSuccessNotification($order));
                            Log::info('Email payment notification sent', [
                                'order_id' => $order->id,
                                'invoice' => $order->invoice_number,
                                'email' => $order->user->email,
                            ]);
                        } catch (\Exception $e) {
                            // Log error but don't fail webhook
                            Log::error('Failed to send email notification: ' . $e->getMessage(), [
                                'order_id' => $order->id,
                                'email' => $order->user->email ?? 'N/A',
                            ]);
                        }
                    }
                }

                return response()->json([
                    'status' => 'success',
                    'message' => 'Webhook processed successfully',
                    'order_id' => $payment->order->invoice_number,
                ], 200);
            }

            return response()->json([
                'status' => 'error',
                'message' => 'Payment not found',
            ], 404);
        } catch (\Exception $e) {
            Log::error('Midtrans Webhook error: ' . $e->getMessage(), [
                'exception' => $e,
                'payload' => $request->all(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Webhook processing failed',
            ], 500);
        }
    }
}



