<?php

namespace App\Http\Controllers\Cashier;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class CashierDashboardController extends Controller
{
    public function __invoke(): View
    {
        $todayStart = Carbon::now()->startOfDay();
        $todayEnd = Carbon::now()->endOfDay();

        $todayOrders = Order::query()
            ->where('channel', 'offline')
            ->whereBetween('created_at', [$todayStart, $todayEnd])
            ->latest()
            ->get();

        // Tidak perlu menampilkan pesanan menunggu di kasir (dipaksa dibayar)
        $waitingPayments = collect();

        $todayIncome = $todayOrders->where('payment_status', 'paid')->sum('grand_total');

        $products = Product::active()
            ->with('category')
            ->select('id', 'category_id', 'name', 'sku', 'price', 'discount_percent', 'stock', 'thumbnail')
            ->orderBy('name')
            ->get()
            ->map(function (Product $product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'price' => $product->final_price,
                    'raw_price' => $product->price,
                    'discount_percent' => $product->discount_percent,
                    'stock' => $product->stock,
                    'thumbnail' => $product->thumbnail,
                    'category_id' => $product->category_id,
                    'category_key' => (string) ($product->category_id ?? 'uncategorized'),
                    'category_name' => $product->category?->name,
                ];
            })
            ->values();

        $categories = $products
            ->groupBy('category_id')
            ->map(function ($items, $id) {
                $name = $items->first()['category_name'] ?? 'Tanpa Kategori';
                return [
                    'id' => (string) ($id ?? 'uncategorized'),
                    'name' => $name,
                    'count' => $items->count(),
                ];
            })
            ->values();

        // Ensure categories is always an array
        if ($categories->isEmpty()) {
            $categories = collect([]);
        }

        $inventorySummary = [
            'totalProducts' => $products->count(),
            'totalStock' => $products->sum('stock'),
            'lowStock' => $products->filter(fn ($product) => $product['stock'] <= 10)->count(),
        ];

        // Share data ke layout untuk cashier view
        view()->share([
            'cashierProducts' => $products,
            'cashierCategories' => $categories,
            'cashierInventory' => $inventorySummary,
        ]);

        return view('cashier.dashboard', [
            'todayOrders' => $todayOrders,
            'waitingPayments' => $waitingPayments,
            'todayIncome' => $todayIncome,
            'products' => $products,
            'categories' => $categories,
            'inventorySummary' => $inventorySummary,
        ]);
    }
}
