<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Enums\UserRole;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View|RedirectResponse
    {
        // Jika sudah login, arahkan sesuai role
        if (Auth::check()) {
            $user = Auth::user();
            if ($user->role === UserRole::Admin) {
                return redirect()->route('admin.dashboard');
            }
            if ($user->role === UserRole::Cashier) {
                return redirect()->route('cashier.dashboard');
            }
            return redirect()->route('home');
        }

        return view('auth.login');
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate('web');

        $request->session()->regenerate();

        $user = Auth::user();

        // Tolak login admin/kasir lewat halaman user, arahkan ke login khusus
        if ($user->role === UserRole::Admin) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            return redirect()
                ->route('admin.login')
                ->withErrors(['email' => 'Silakan login via halaman admin.']);
        }

        if ($user->role === UserRole::Cashier) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            return redirect()
                ->route('cashier.login')
                ->withErrors(['email' => 'Silakan login via halaman kasir.']);
        }

        // Pelanggan diarahkan ke beranda
        return redirect()->route('home');
    }

    /**
     * Destroy an authenticated session.
     * Only logout the guard matching the current URL path to allow
     * simultaneous login for admin/cashier/user in different tabs.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $path = '/' . ltrim($request->path(), '/');
        $referer = $request->headers->get('referer', '');

        // Determine which guard to logout based on URL path or referer
        if (str_starts_with($path, '/admin') || str_contains($referer, '/admin')) {
            Auth::guard('admin')->logout();
            // Only invalidate session for this specific cookie
            return redirect()->route('admin.login')->with('status', 'Anda telah keluar dari panel admin.');
        }

        if (str_starts_with($path, '/kasir') || str_contains($referer, '/kasir')) {
            Auth::guard('cashier')->logout();
            return redirect()->route('cashier.login')->with('status', 'Anda telah keluar dari panel kasir.');
        }

        // Default: logout web guard only (for regular customers)
        Auth::guard('web')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('status', 'Anda telah keluar.');
    }
}
