<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class AdminProductController extends Controller
{
    public function index(Request $request): View
    {
        $products = Product::query()
            ->with('category')
            ->latest()
            ->paginate(12);

        return view('admin.products.index', [
            'products' => $products,
        ]);
    }

    public function create(): View
    {
        return view('admin.products.create', [
            'categories' => Category::active()->get(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $this->validateProduct($request);

        $validated['is_featured'] = $request->boolean('is_featured');
        $validated['min_order_qty'] = $validated['min_order_qty'] ?? 1;

        if ($request->hasFile('thumbnail')) {
            $validated['thumbnail'] = $request->file('thumbnail')->store('products', 'public');
        }

        $product = Product::create(array_merge($validated, [
            'created_by' => $request->user()->id,
        ]));

        if ($request->hasFile('gallery')) {
            foreach ($request->file('gallery') as $index => $image) {
                $product->images()->create([
                    'path' => $image->store('products', 'public'),
                    'alt_text' => $product->name,
                    'display_order' => $index + 1,
                    'is_primary' => $index === 0,
                ]);
            }
        }

        return redirect()->route('admin.products.index')->with('status', 'Produk berhasil ditambahkan.');
    }

    public function edit(Product $product): View
    {
        $product->load('images');

        return view('admin.products.edit', [
            'product' => $product,
            'categories' => Category::active()->get(),
        ]);
    }

    public function update(Request $request, Product $product)
    {
        $validated = $this->validateProduct($request, $product->id);

        $validated['is_featured'] = $request->boolean('is_featured');
        $validated['min_order_qty'] = $validated['min_order_qty'] ?? 1;

        if ($request->hasFile('thumbnail')) {
            if ($product->getRawOriginal('thumbnail')) {
                Storage::disk('public')->delete($product->getRawOriginal('thumbnail'));
            }

            $validated['thumbnail'] = $request->file('thumbnail')->store('products', 'public');
        } elseif ($request->boolean('remove_thumbnail')) {
            if ($product->getRawOriginal('thumbnail')) {
                Storage::disk('public')->delete($product->getRawOriginal('thumbnail'));
            }
            $validated['thumbnail'] = null;
        }

        $product->update($validated);

        if ($request->hasFile('gallery')) {
            foreach ($request->file('gallery') as $index => $image) {
                $product->images()->create([
                    'path' => $image->store('products', 'public'),
                    'alt_text' => $product->name,
                    'display_order' => $product->images()->count() + $index + 1,
                    'is_primary' => false,
                ]);
            }
        }

        if ($request->filled('remove_images')) {
            $product->images()
                ->whereIn('id', $request->input('remove_images', []))
                ->get()
                ->each(function ($image) {
                    Storage::disk('public')->delete($image->getRawOriginal('path'));
                    $image->delete();
                });
        }

        return redirect()->route('admin.products.index')->with('status', 'Produk berhasil diperbarui.');
    }

    public function destroy(Product $product)
    {
        if ($product->getRawOriginal('thumbnail')) {
            Storage::disk('public')->delete($product->getRawOriginal('thumbnail'));
        }

        $product->images->each(function ($image) {
            Storage::disk('public')->delete($image->getRawOriginal('path'));
            $image->delete();
        });

        $product->delete();

        return redirect()->route('admin.products.index')->with('status', 'Produk berhasil dihapus.');
    }

    protected function validateProduct(Request $request, ?int $productId = null): array
    {
        return $request->validate([
            'category_id' => ['required', 'exists:categories,id'],
            'sku' => ['required', 'string', 'max:30', Rule::unique('products', 'sku')->ignore($productId)],
            'name' => ['required', 'string', 'max:150'],
            'tagline' => ['nullable', 'string', 'max:160'],
            'summary' => ['nullable', 'string'],
            'description' => ['nullable', 'string'],
            'stock' => ['required', 'integer', 'min:0'],
            'min_order_qty' => ['nullable', 'integer', 'min:1'],
            'max_order_qty' => ['nullable', 'integer', 'min:1'],
            'weight_grams' => ['nullable', 'integer', 'min:0'],
            'price' => ['required', 'integer', 'min:0'],
            'cost_price' => ['nullable', 'integer', 'min:0'],
            'discount_percent' => ['nullable', 'numeric', 'min:0', 'max:90'],
            'status' => ['required', Rule::in(['draft', 'active', 'archived'])],
            'is_featured' => ['nullable', 'boolean'],
            'thumbnail' => ['nullable', 'image', 'max:2048'],
            'gallery.*' => ['nullable', 'image', 'max:2048'],
            'metadata' => ['nullable', 'array'],
        ]);
    }
}
