<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class AdminOrderController extends Controller
{
    public function index(Request $request): View
    {
        $filters = [
            'search' => $request->string('search')->toString(),
            'channel' => $request->string('channel')->toString(),
            'status' => $request->string('status')->toString(),
            'payment_status' => $request->string('payment_status')->toString(),
        ];

        $ordersQuery = Order::with(['user', 'cashier'])
            ->when($filters['search'], function ($query, $search) {
                $query->where(function ($builder) use ($search) {
                    $builder
                        ->where('invoice_number', 'like', "%{$search}%")
                        ->orWhere('metadata->customer_name', 'like', "%{$search}%")
                        ->orWhere('shipping_address->recipient_name', 'like', "%{$search}%")
                        ->orWhere('shipping_address->phone', 'like', "%{$search}%");
                });
            })
            ->when($filters['channel'] && $filters['channel'] !== 'all', fn ($query) => $query->where('channel', $filters['channel']))
            ->when($filters['status'] && $filters['status'] !== 'all', fn ($query) => $query->where('status', $filters['status']))
            ->when($filters['payment_status'] && $filters['payment_status'] !== 'all', fn ($query) => $query->where('payment_status', $filters['payment_status']))
            ->orderByDesc('created_at');

        $orders = $ordersQuery->paginate(12)->withQueryString();

        $today = Carbon::today();

        $summary = [
            'todayOnline' => Order::where('channel', 'online')->whereBetween('created_at', [$today->copy()->startOfDay(), $today->copy()->endOfDay()])->count(),
            'todayOffline' => Order::where('channel', 'offline')->whereBetween('created_at', [$today->copy()->startOfDay(), $today->copy()->endOfDay()])->count(),
            'totalOrders' => Order::where('channel', 'online')->count(),
            'awaitingPayment' => Order::where('payment_status', '!=', 'paid')->count(),
            'totalIncome' => Order::where('payment_status', 'paid')->sum('grand_total'),
        ];

        return view('admin.orders.index', [
            'orders' => $orders,
            'filters' => $filters,
            'summary' => $summary,
            'statusOptions' => [
                'all' => 'Semua Status',
                'pending' => 'Pending',
                'confirmed' => 'Confirmed',
                'processing' => 'Processing',
                'shipped' => 'Shipped',
                'completed' => 'Completed',
                'cancelled' => 'Cancelled',
                'refunded' => 'Refunded',
            ],
            'paymentStatusOptions' => [
                'all' => 'Semua Pembayaran',
                'unpaid' => 'Belum Dibayar',
                'awaiting_confirmation' => 'Menunggu Konfirmasi',
                'paid' => 'Sudah Dibayar',
                'failed' => 'Gagal',
                'refunded' => 'Refund',
            ],
            'channelOptions' => [
                'all' => 'Semua Kanal',
                'online' => 'Online Store',
                'offline' => 'POS Kasir',
            ],
        ]);
    }

    public function show(Order $order): View
    {
        $order->load(['items.product.images', 'payments', 'cashier', 'user']);

        return view('admin.orders.show', [
            'order' => $order,
        ]);
    }

    public function print(Order $order): View
    {
        $order->load(['items.product', 'payments', 'cashier']);

        return view('admin.orders.print', [
            'order' => $order,
            'storeName' => config('app.name', 'D\'Shop'),
        ]);
    }

    public function updateStatus(Request $request, Order $order): RedirectResponse
    {
        $validated = $request->validate([
            'status' => 'required|string|in:pending,confirmed,processing,shipped,completed,cancelled,refunded',
            'tracking_number' => 'nullable|string|max:255',
        ]);

        $order->update([
            'status' => $validated['status'],
            'tracking_number' => $validated['tracking_number'] ?? $order->tracking_number,
        ]);

        return redirect()
            ->route('admin.orders.detail', $order)
            ->with('success', 'Status pesanan berhasil diperbarui!');
    }

    public function shipOrder(Request $request, Order $order): RedirectResponse
    {
        $validated = $request->validate([
            'shipping_courier' => 'nullable|string|max:50',
            'tracking_number' => 'nullable|string|max:255',
        ]);

        $order->update([
            'status' => 'shipped',
            'shipping_courier' => $validated['shipping_courier'] ?? $order->shipping_courier,
            'tracking_number' => $validated['tracking_number'] ?? $order->tracking_number,
        ]);

        return redirect()
            ->route('admin.orders.detail', $order)
            ->with('success', 'Pesanan ditandai sebagai dikirim!');
    }
}
