<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Journal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class AdminJournalController extends Controller
{
    public function index(): View
    {
        $journals = Journal::query()
            ->latest('published_at')
            ->paginate(12);

        return view('admin.journals.index', [
            'journals' => $journals,
        ]);
    }

    public function create(): View
    {
        return view('admin.journals.create');
    }

    public function store(Request $request)
    {
        $validated = $this->validateJournal($request);

        $validated['is_featured'] = $request->boolean('is_featured');

        if ($request->hasFile('hero_image')) {
            $validated['hero_image'] = $request->file('hero_image')->store('journals', 'public');
        }

        $validated['slug'] = Str::slug($validated['title']);
        $validated['tags'] = $this->parseTags($request->input('tags'));
        $validated['author_id'] = $request->user()->id;

        Journal::create($validated);

        return redirect()->route('admin.journals.index')->with('status', 'Jurnal berhasil ditambahkan.');
    }

    public function edit(Journal $journal): View
    {
        return view('admin.journals.edit', [
            'journal' => $journal,
        ]);
    }

    public function update(Request $request, Journal $journal)
    {
        $validated = $this->validateJournal($request, $journal->id);

        $validated['is_featured'] = $request->boolean('is_featured');

        if ($request->hasFile('hero_image')) {
            if ($journal->hero_image) {
                Storage::disk('public')->delete($journal->hero_image);
            }

            $validated['hero_image'] = $request->file('hero_image')->store('journals', 'public');
        }

        $validated['tags'] = $this->parseTags($request->input('tags'));
        $validated['author_id'] = $request->user()->id;

        $journal->update($validated);

        return redirect()->route('admin.journals.edit', $journal)->with('status', 'Jurnal berhasil diperbarui.');
    }

    public function destroy(Journal $journal)
    {
        if ($journal->hero_image) {
            Storage::disk('public')->delete($journal->hero_image);
        }

        $journal->delete();

        return redirect()->route('admin.journals.index')->with('status', 'Jurnal berhasil dihapus.');
    }

    protected function validateJournal(Request $request, ?int $journalId = null): array
    {
        return $request->validate([
            'title' => ['required', 'string', 'max:160'],
            'subtitle' => ['nullable', 'string', 'max:160'],
            'excerpt' => ['nullable', 'string'],
            'body' => ['required', 'string'],
            'tags' => ['nullable', 'string'],
            'published_at' => ['nullable', 'date'],
            'is_featured' => ['nullable', 'boolean'],
            'hero_image' => ['nullable', 'image', 'max:2048'],
            'slug' => ['nullable', Rule::unique('journals', 'slug')->ignore($journalId)],
        ]);
    }

    protected function parseTags(?string $tags): array
    {
        if (! $tags) {
            return [];
        }

        return collect(explode(',', $tags))
            ->map(fn ($tag) => trim($tag))
            ->filter()
            ->unique()
            ->values()
            ->all();
    }
}
