<?php

namespace App\Http\Controllers\Admin;

use App\Exports\IncomeExport;
use App\Http\Controllers\Controller;
use App\Models\Order;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Date;
use Maatwebsite\Excel\Facades\Excel;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class AdminIncomeController extends Controller
{
    public function index(Request $request)
    {
        [$start, $end] = $this->getDateRange($request);

        $orders = Order::query()
            ->where('payment_status', 'paid')
            ->whereBetween('paid_at', [$start->startOfDay(), $end->endOfDay()])
            ->with('items')
            ->orderByDesc('paid_at')
            ->paginate(15)
            ->withQueryString();

        $totalIncome = $orders->getCollection()->sum(fn (Order $order) => $order->net_income);

        return view('admin.income.index', [
            'orders' => $orders,
            'totalIncome' => $totalIncome,
            'start' => $start->toDateString(),
            'end' => $end->toDateString(),
        ]);
    }

    public function export(Request $request): BinaryFileResponse
    {
        [$start, $end] = $this->getDateRange($request);

        $orders = Order::query()
            ->where('payment_status', 'paid')
            ->whereBetween('paid_at', [$start->startOfDay(), $end->endOfDay()])
            ->with('items')
            ->orderByDesc('paid_at')
            ->get();

        $fileName = 'penghasilan-' . $start->format('Ymd') . '-' . $end->format('Ymd') . '.xlsx';

        return Excel::download(
            new IncomeExport($orders, "{$start->translatedFormat('d M Y')} - {$end->translatedFormat('d M Y')}"),
            $fileName
        );
    }

    protected function getDateRange(Request $request): array
    {
        $start = $request->filled('start')
            ? Carbon::parse($request->input('start'))
            : now()->startOfMonth();

        $end = $request->filled('end')
            ? Carbon::parse($request->input('end'))
            : now()->endOfMonth();

        if ($start->greaterThan($end)) {
            [$start, $end] = [$end, $start];
        }

        return [$start, $end];
    }
}
