<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\View\View;

class AdminDashboardController extends Controller
{
    public function __invoke(): View
    {
        $paidOrders = Order::query()->where('payment_status', 'paid');

        $monthExpression = $this->monthExtractExpression('paid_at');

        $totalIncome = (clone $paidOrders)->sum('grand_total');
        $monthlyIncome = (clone $paidOrders)
            ->whereBetween('paid_at', [now()->startOfMonth(), now()->endOfMonth()])
            ->sum('grand_total');

        $incomeTrend = (clone $paidOrders)
            ->selectRaw("{$monthExpression} as month_key, SUM(grand_total) as total")
            ->groupBy('month_key')
            ->orderBy('month_key', 'asc')
            ->take(6)
            ->get();

        $incomeTrend = $incomeTrend
            ->filter(fn ($row) => ! empty($row->month_key))
            ->map(function ($row) {
            $month = $row->month_key;
            $date = Carbon::createFromFormat('Y-m', $month)->startOfMonth();

            return (object) [
                'month' => $month,
                'label' => $date->translatedFormat('M Y'),
                'total' => $row->total,
            ];
            })
            ->values();

        $bestSellers = Product::query()
            ->select('products.*')
            ->withCount(['orderItems as sold_quantity' => function ($query) {
                $query->selectRaw('COALESCE(SUM(quantity),0)');
            }])
            ->orderByDesc('sold_quantity')
            ->take(5)
            ->get();

        $waitingOrders = Order::query()
            ->whereIn('status', ['pending', 'confirmed'])
            ->latest()
            ->take(5)
            ->get();

        $totalOrders = Order::query()->count();
        $onlineOrders = Order::query()->where('channel', 'online')->count();
        $offlineOrders = Order::query()->where('channel', 'offline')->count();
        $unpaidOrders = Order::query()->where('payment_status', '!=', 'paid')->count();

        $today = Carbon::today();
        $todayIncome = (clone $paidOrders)
            ->whereBetween('paid_at', [$today->startOfDay(), $today->endOfDay()])
            ->sum('grand_total');

        $incomeChartLabels = $incomeTrend->pluck('label');
        $incomeChartTotals = $incomeTrend->pluck('total');

        $channelChart = [
            'labels' => ['Online Store', 'POS Kasir', 'Belum Dibayar'],
            'values' => [$onlineOrders, $offlineOrders, $unpaidOrders],
        ];

        $topProductsChart = [
            'labels' => $bestSellers->pluck('name'),
            'values' => $bestSellers->pluck('sold_quantity'),
        ];

        $userCount = User::query()->count();

        // Data penjualan lengkap
        $totalSales = Order::query()->where('payment_status', 'paid')->count();
        $todaySales = Order::query()
            ->where('payment_status', 'paid')
            ->whereBetween('paid_at', [$today->startOfDay(), $today->endOfDay()])
            ->count();
        
        // Stok per produk
        $lowStockProducts = Product::query()
            ->where('status', 'active')
            ->where('stock', '<=', 10)
            ->orderBy('stock', 'asc')
            ->take(10)
            ->get();
        
        $totalProducts = Product::query()->where('status', 'active')->count();
        $totalStock = Product::query()->where('status', 'active')->sum('stock');
        $outOfStockProducts = Product::query()
            ->where('status', 'active')
            ->where('stock', '<=', 0)
            ->count();

        // Data penjualan per produk dengan nama
        $salesByProduct = Product::query()
            ->select('products.*')
            ->withCount(['orderItems as sold_quantity' => function ($query) {
                $query->selectRaw('COALESCE(SUM(quantity),0)')
                    ->whereHas('order', function ($q) {
                        $q->where('payment_status', 'paid');
                    });
            }])
            ->withSum(['orderItems as revenue' => function ($query) {
                $query->selectRaw('COALESCE(SUM(subtotal),0)')
                    ->whereHas('order', function ($q) {
                        $q->where('payment_status', 'paid');
                    });
            }], 'subtotal')
            ->orderByDesc('sold_quantity')
            ->take(10)
            ->get();

        return view('admin.dashboard', [
            'totalIncome' => $totalIncome,
            'monthlyIncome' => $monthlyIncome,
            'todayIncome' => $todayIncome,
            'incomeTrend' => $incomeTrend,
            'bestSellers' => $bestSellers,
            'waitingOrders' => $waitingOrders,
            'totalOrders' => $totalOrders,
            'onlineOrders' => $onlineOrders,
            'offlineOrders' => $offlineOrders,
            'unpaidOrders' => $unpaidOrders,
            'incomeChartLabels' => $incomeChartLabels,
            'incomeChartTotals' => $incomeChartTotals,
            'channelChart' => $channelChart,
            'topProductsChart' => $topProductsChart,
            'userCount' => $userCount,
            'totalSales' => $totalSales,
            'todaySales' => $todaySales,
            'lowStockProducts' => $lowStockProducts,
            'totalProducts' => $totalProducts,
            'totalStock' => $totalStock,
            'outOfStockProducts' => $outOfStockProducts,
            'salesByProduct' => $salesByProduct,
        ]);
    }

    protected function monthExtractExpression(string $column): string
    {
        $driver = Order::query()->getModel()->getConnection()->getDriverName();

        return match ($driver) {
            'sqlite' => "strftime('%Y-%m', {$column})",
            'pgsql' => "to_char({$column}, 'YYYY-MM')",
            'sqlsrv' => "FORMAT({$column}, 'yyyy-MM')",
            default => "DATE_FORMAT({$column}, '%Y-%m')",
        };
    }
}
