<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\View\View;

class AdminCategoryController extends Controller
{
    public function index(Request $request): View
    {
        $filters = [
            'search' => $request->string('search')->toString(),
            'status' => $request->string('status')->toString(),
        ];

        $categoriesQuery = Category::query()
            ->when($filters['search'], function ($query, $search) {
                $query->where(function ($builder) use ($search) {
                    $builder
                        ->where('name', 'like', "%{$search}%")
                        ->orWhere('tagline', 'like', "%{$search}%")
                        ->orWhere('slug', 'like', "%{$search}%");
                });
            })
            ->when($filters['status'] && $filters['status'] !== 'all', function ($query) use ($filters) {
                $query->where('is_active', $filters['status'] === 'active');
            })
            ->orderBy('display_order')
            ->orderBy('name');

        $categories = $categoriesQuery->paginate(12)->withQueryString();

        $stats = [
            'total' => Category::count(),
            'active' => Category::where('is_active', true)->count(),
            'inactive' => Category::where('is_active', false)->count(),
        ];

        return view('admin.categories.index', [
            'categories' => $categories,
            'filters' => $filters,
            'stats' => $stats,
        ]);
    }

    public function create(): View
    {
        return view('admin.categories.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'tagline' => ['nullable', 'string', 'max:160'],
            'description' => ['nullable', 'string', 'max:1000'],
            'hero_image' => ['nullable', 'image', 'max:2048'],
            'hero_image_url' => ['nullable', 'url', 'max:500'],
            'carousel_image' => ['nullable', 'image', 'max:2048'],
            'carousel_image_url' => ['nullable', 'url', 'max:500'],
            'display_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $slug = Str::slug($data['name']);
        $originalSlug = $slug;
        $counter = 1;

        while (Category::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $data['slug'] = $slug;
        $data['is_active'] = $request->boolean('is_active');
        $data['display_order'] = $data['display_order'] ?? 0;

        DB::transaction(function () use ($request, &$data) {
            if ($request->hasFile('hero_image')) {
                $data['hero_image'] = $request->file('hero_image')->store('categories', 'public');
            } elseif ($request->filled('hero_image_url')) {
                $data['hero_image'] = $request->hero_image_url;
            }

            if ($request->hasFile('carousel_image')) {
                $data['carousel_image'] = $request->file('carousel_image')->store('categories', 'public');
            } elseif ($request->filled('carousel_image_url')) {
                $data['carousel_image'] = $request->carousel_image_url;
            }

            Category::create($data);
        });

        return redirect()->route('admin.categories.index')->with('status', 'Kategori berhasil ditambahkan.');
    }

    public function edit(Category $category): View
    {
        return view('admin.categories.edit', [
            'category' => $category,
        ]);
    }

    public function update(Request $request, Category $category)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'tagline' => ['nullable', 'string', 'max:160'],
            'description' => ['nullable', 'string', 'max:1000'],
            'hero_image' => ['nullable', 'image', 'max:2048'],
            'hero_image_url' => ['nullable', 'url', 'max:500'],
            'carousel_image' => ['nullable', 'image', 'max:2048'],
            'carousel_image_url' => ['nullable', 'url', 'max:500'],
            'display_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $slug = Str::slug($data['name']);
        if ($slug !== $category->slug) {
            $originalSlug = $slug;
            $counter = 1;
            while (Category::where('slug', $slug)->where('id', '!=', $category->id)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }
        }
        $data['slug'] = $slug;
        $data['is_active'] = $request->boolean('is_active');
        $data['display_order'] = $data['display_order'] ?? 0;

        DB::transaction(function () use ($request, &$data, $category) {
            if ($request->hasFile('hero_image')) {
                $this->deleteStoredImage($category->getRawOriginal('hero_image'));
                $data['hero_image'] = $request->file('hero_image')->store('categories', 'public');
            } elseif ($request->filled('hero_image_url')) {
                $this->deleteStoredImage($category->getRawOriginal('hero_image'));
                $data['hero_image'] = $request->hero_image_url;
            }

            if ($request->hasFile('carousel_image')) {
                $this->deleteStoredImage($category->getRawOriginal('carousel_image'));
                $data['carousel_image'] = $request->file('carousel_image')->store('categories', 'public');
            } elseif ($request->filled('carousel_image_url')) {
                $this->deleteStoredImage($category->getRawOriginal('carousel_image'));
                $data['carousel_image'] = $request->carousel_image_url;
            }

            $category->update($data);
        });

        return redirect()->route('admin.categories.index')->with('status', 'Kategori berhasil diperbarui.');
    }

    protected function deleteStoredImage(?string $path): void
    {
        if (empty($path)) {
            return;
        }

        if (Str::startsWith($path, ['http://', 'https://', 'data:'])) {
            return;
        }

        Storage::disk('public')->delete($path);
    }

    public function destroy(Category $category)
    {
        try {
            $productCount = $category->products()->withTrashed()->count();
            
            if ($productCount > 0) {
                return back()->withErrors(['category' => "Kategori tidak dapat dihapus karena masih memiliki {$productCount} produk. Hapus atau pindahkan produk terlebih dahulu."]);
            }

            DB::transaction(function () use ($category) {
                $this->deleteStoredImage($category->getRawOriginal('hero_image'));
                $this->deleteStoredImage($category->getRawOriginal('carousel_image'));
                $category->delete();
            });

            return redirect()->route('admin.categories.index')->with('status', 'Kategori berhasil dihapus.');
        } catch (\Illuminate\Database\QueryException $e) {
            if ($e->getCode() === '23000') {
                return back()->withErrors(['category' => 'Kategori tidak dapat dihapus karena masih memiliki produk yang terkait.']);
            }
            return back()->withErrors(['category' => 'Terjadi kesalahan saat menghapus kategori: ' . $e->getMessage()]);
        } catch (\Exception $e) {
            return back()->withErrors(['category' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }
}
