<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\View\View;

class AdminCashierReportController extends Controller
{
    public function index(): View
    {
        // Statistik Transaksi Kasir Hari Ini
        $today = Carbon::today();
        $todayOfflineOrders = Order::where('channel', 'offline')
            ->whereDate('created_at', $today)
            ->count();
        $todayOfflineRevenue = Order::where('channel', 'offline')
            ->where('payment_status', 'paid')
            ->whereDate('paid_at', $today)
            ->sum('grand_total');

        // Statistik Transaksi Kasir Bulan Ini
        $monthStart = now()->startOfMonth();
        $monthEnd = now()->endOfMonth();
        $monthOfflineOrders = Order::where('channel', 'offline')
            ->whereBetween('created_at', [$monthStart, $monthEnd])
            ->count();
        $monthOfflineRevenue = Order::where('channel', 'offline')
            ->where('payment_status', 'paid')
            ->whereBetween('paid_at', [$monthStart, $monthEnd])
            ->sum('grand_total');

        // Total Transaksi Kasir
        $totalOfflineOrders = Order::where('channel', 'offline')->count();
        $totalOfflineRevenue = Order::where('channel', 'offline')
            ->where('payment_status', 'paid')
            ->sum('grand_total');

        // Rata-rata transaksi per hari (bulan ini)
        $daysInMonth = now()->daysInMonth;
        $avgDailyOrders = $daysInMonth > 0 ? round($monthOfflineOrders / $daysInMonth, 1) : 0;
        $avgDailyRevenue = $daysInMonth > 0 ? round($monthOfflineRevenue / $daysInMonth) : 0;

        // Transaksi Terbaru (7 hari terakhir)
        $recentOrders = Order::where('channel', 'offline')
            ->with(['cashier', 'items.product'])
            ->latest()
            ->take(10)
            ->get();

        // Statistik per Kasir
        $cashierStats = User::whereIn('id', function ($query) {
                $query->select('cashier_id')
                    ->from('orders')
                    ->where('channel', 'offline')
                    ->whereNotNull('cashier_id');
            })
            ->get()
            ->map(function ($user) {
                $orders = Order::where('cashier_id', $user->id)
                    ->where('channel', 'offline')
                    ->get();

                return (object) [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'offline_orders_count' => $orders->count(),
                    'offline_revenue' => $orders->where('payment_status', 'paid')->sum('grand_total'),
                ];
            })
            ->filter(fn($stat) => $stat->offline_orders_count > 0)
            ->sortByDesc('offline_revenue')
            ->take(5);

        // Grafik transaksi harian (7 hari terakhir)
        $dailyStats = collect(range(6, 0))->map(function ($daysAgo) {
            $date = Carbon::today()->subDays($daysAgo);
            $orders = Order::where('channel', 'offline')
                ->whereDate('created_at', $date)
                ->count();
            $revenue = Order::where('channel', 'offline')
                ->where('payment_status', 'paid')
                ->whereDate('paid_at', $date)
                ->sum('grand_total');

            return [
                'date' => $date->format('Y-m-d'),
                'label' => $date->format('d M'),
                'day' => $date->format('D'),
                'orders' => $orders,
                'revenue' => $revenue,
            ];
        });

        // Produk Terlaris di Kasir
        $topProducts = \App\Models\Product::query()
            ->select([
                'products.id',
                'products.name',
                'products.price',
            ])
            ->join('order_items', 'products.id', '=', 'order_items.product_id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.channel', 'offline')
            ->selectRaw('SUM(order_items.quantity) as total_sold')
            ->groupBy('products.id', 'products.name', 'products.price')
            ->orderByDesc('total_sold')
            ->take(10)
            ->get();

        // Status Pembayaran
        $paymentStatusStats = [
            'paid' => Order::where('channel', 'offline')
                ->where('payment_status', 'paid')
                ->count(),
            'unpaid' => Order::where('channel', 'offline')
                ->where('payment_status', 'unpaid')
                ->count(),
            'awaiting_confirmation' => Order::where('channel', 'offline')
                ->where('payment_status', 'awaiting_confirmation')
                ->count(),
        ];

        return view('admin.cashier-report', [
            'todayOfflineOrders' => $todayOfflineOrders,
            'todayOfflineRevenue' => $todayOfflineRevenue,
            'monthOfflineOrders' => $monthOfflineOrders,
            'monthOfflineRevenue' => $monthOfflineRevenue,
            'totalOfflineOrders' => $totalOfflineOrders,
            'totalOfflineRevenue' => $totalOfflineRevenue,
            'avgDailyOrders' => $avgDailyOrders,
            'avgDailyRevenue' => $avgDailyRevenue,
            'recentOrders' => $recentOrders,
            'cashierStats' => $cashierStats,
            'dailyStats' => $dailyStats,
            'topProducts' => $topProducts,
            'paymentStatusStats' => $paymentStatusStats,
        ]);
    }
}
