<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Order;
use App\Models\Product;
use Illuminate\View\View;

class AdminAnalysisController extends Controller
{
    public function index(): View
    {
        $totalOrders = Order::query()->count();
        $paidRevenue = Order::query()
            ->where('payment_status', 'paid')
            ->sum('grand_total');

        $onlineOrders = Order::query()->where('channel', 'online')->count();
        $offlineOrders = Order::query()->where('channel', 'offline')->count();
        $unpaidOrders = Order::query()->where('payment_status', '!=', 'paid')->count();

        $paidOrdersQuery = Order::query()->where('payment_status', 'paid');
        $onlinePaidOrdersQuery = (clone $paidOrdersQuery)->where('channel', 'online');
        $offlinePaidOrdersQuery = (clone $paidOrdersQuery)->where('channel', 'offline');

        $onlineRevenue = (clone $onlinePaidOrdersQuery)->sum('grand_total');
        $offlineRevenue = (clone $offlinePaidOrdersQuery)->sum('grand_total');

        $onlineRevenueShare = $paidRevenue > 0 ? round(($onlineRevenue / $paidRevenue) * 100) : 0;
        $offlineRevenueShare = $paidRevenue > 0 ? round(($offlineRevenue / $paidRevenue) * 100) : 0;

        $totalChannelOrders = max($onlineOrders + $offlineOrders, 1);
        $onlineOrderShare = round(($onlineOrders / $totalChannelOrders) * 100);
        $offlineOrderShare = round(($offlineOrders / $totalChannelOrders) * 100);

        $totalCategories = Category::query()->count();
        $activeCategories = Category::query()->where('is_active', true)->count();
        $inactiveCategories = max(0, $totalCategories - $activeCategories);

        $activeProducts = Product::query()->where('status', 'active')->count();

        $statusChart = [
            'labels' => ['Aktif', 'Nonaktif'],
            'data' => [$activeCategories, $inactiveCategories],
        ];

        $channelChart = [
            'labels' => ['Online Store', 'POS Kasir', 'Belum Dibayar'],
            'data' => [$onlineOrders, $offlineOrders, $unpaidOrders],
        ];

        $topProducts = Product::query()
            ->select('products.*')
            ->withCount(['orderItems as sold_quantity' => function ($query) {
                $query->selectRaw('COALESCE(SUM(quantity),0)');
            }])
            ->orderByDesc('sold_quantity')
            ->limit(6)
            ->get();

        $topProductChart = [
            'labels' => $topProducts->pluck('name')->toArray(),
            'data' => $topProducts->pluck('sold_quantity')->toArray(),
        ];

        $channelRevenueChart = [
            'labels' => ['Online Store', 'POS Kasir'],
            'data' => [$onlineRevenue, $offlineRevenue],
        ];

        return view('admin.analysis', [
            'totalOrders' => $totalOrders,
            'paidRevenue' => $paidRevenue,
            'activeProducts' => $activeProducts,
            'totalCategories' => $totalCategories,
            'activeCategories' => $activeCategories,
            'statusChart' => $statusChart,
            'channelChart' => $channelChart,
            'topProductChart' => $topProductChart,
            'topProducts' => $topProducts,
            'onlineOrders' => $onlineOrders,
            'offlineOrders' => $offlineOrders,
            'onlineRevenue' => $onlineRevenue,
            'offlineRevenue' => $offlineRevenue,
            'onlineOrderShare' => $onlineOrderShare,
            'offlineOrderShare' => $offlineOrderShare,
            'onlineRevenueShare' => $onlineRevenueShare,
            'offlineRevenueShare' => $offlineRevenueShare,
            'channelRevenueChart' => $channelRevenueChart,
        ]);
    }
}
