<?php

namespace App\Controllers;

use App\Models\ActivityModel;
use CodeIgniter\Exceptions\PageNotFoundException;
use CodeIgniter\HTTP\Files\UploadedFile;

class ActivityAdmin extends BaseController
{
    private ActivityModel $activities;

    public function __construct()
    {
        helper(['text']);
        $this->activities = new ActivityModel();
    }

    public function index(): string
    {
        $search     = trim((string) $this->request->getGet('q'));
        $mediaType  = (string) $this->request->getGet('media');
        $dateFilter = (string) $this->request->getGet('date');
        $sort       = (string) $this->request->getGet('sort');

        $query = $this->activities;

        if ($search !== '') {
            $query->groupStart()
                ->like('title', $search)
                ->orLike('description', $search)
                ->orLike('location', $search)
                ->groupEnd();
        }

        if (in_array($mediaType, ['image', 'video', 'none'], true)) {
            $query->where('media_type', $mediaType);
        }

        if ($dateFilter !== '') {
            if (preg_match('/^\\d{4}-\\d{2}-\\d{2}$/', $dateFilter)) {
                $query->where('activity_date', $dateFilter);
            } elseif (preg_match('/^\\d{4}-\\d{2}$/', $dateFilter)) {
                $startDate = $dateFilter . '-01';
                $endDate   = date('Y-m-t', strtotime($startDate));
                $query->where('activity_date >=', $startDate)
                    ->where('activity_date <=', $endDate);
            }
        }

        $sort = $sort !== '' ? $sort : 'date_desc';
        switch ($sort) {
            case 'date_asc':
                $query->orderBy('activity_date', 'asc')->orderBy('activity_time', 'asc');
                break;
            case 'name_asc':
                $query->orderBy('title', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('title', 'desc');
                break;
            default:
                $query->orderBy('activity_date', 'desc')->orderBy('activity_time', 'desc');
                $sort = 'date_desc';
        }

        $activities = $query->paginate(10, 'activities');

        return view('admin_list_activities', [
            'activities' => $activities,
            'pager'      => $this->activities->pager,
            'filters'    => [
                'search' => $search,
                'media'  => $mediaType !== '' ? $mediaType : 'semua',
                'date'   => $dateFilter,
                'sort'   => $sort,
            ],
        ]);
    }

    public function create()
    {
        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_activity', [
                    'mode'       => 'create',
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $mediaFile = $this->request->getFile('media');
            $mediaData = $this->storeActivityMedia($mediaFile);

            $payload = [
                'title'         => (string) $this->request->getPost('title'),
                'activity_date' => (string) $this->request->getPost('activity_date'),
                'activity_time' => (string) $this->request->getPost('activity_time') ?: null,
                'location'      => (string) $this->request->getPost('location') ?: null,
                'description'   => (string) $this->request->getPost('description') ?: null,
                'media_type'    => $mediaData['type'] ?? 'none',
                'media_path'    => $mediaData['path'] ?? null,
            ];

            $this->activities->insert($payload);

            return redirect()
                ->to('/admin/aktivitas')
                ->with('message', 'Aktivitas berhasil ditambahkan.');
        }

        return view('admin_form_activity', [
            'mode' => 'create',
        ]);
    }

    public function edit(int $id)
    {
        $activity = $this->findOrFail($id);

        if ($this->request->is('post')) {
            $rules = $this->rules();

            if (! $this->validate($rules)) {
                return view('admin_form_activity', [
                    'mode'       => 'edit',
                    'activity'   => $activity,
                    'validation' => $this->validator,
                    'oldInput'   => $this->request->getPost(),
                ]);
            }

            $mediaFile   = $this->request->getFile('media');
            $mediaData   = $this->storeActivityMedia($mediaFile);
            $removeMedia = (bool) $this->request->getPost('remove_media');

            $payload = [
                'title'         => (string) $this->request->getPost('title'),
                'activity_date' => (string) $this->request->getPost('activity_date'),
                'activity_time' => (string) $this->request->getPost('activity_time') ?: null,
                'location'      => (string) $this->request->getPost('location') ?: null,
                'description'   => (string) $this->request->getPost('description') ?: null,
            ];

            if (! empty($mediaData)) {
                if (! empty($activity['media_path'])) {
                    $this->deleteActivityMedia($activity['media_path']);
                }
                $payload['media_type'] = $mediaData['type'];
                $payload['media_path'] = $mediaData['path'];
            } elseif ($removeMedia) {
                $this->deleteActivityMedia($activity['media_path'] ?? null);
                $payload['media_type'] = 'none';
                $payload['media_path'] = null;
            }

            $this->activities->update($id, $payload);

            return redirect()
                ->to('/admin/aktivitas')
                ->with('message', 'Aktivitas berhasil diperbarui.');
        }

        return view('admin_form_activity', [
            'mode'     => 'edit',
            'activity' => $activity,
        ]);
    }

    public function delete(int $id)
    {
        $activity = $this->findOrFail($id);
        $this->activities->delete($id);

        if (! empty($activity['media_path'])) {
            $this->deleteActivityMedia($activity['media_path']);
        }

        return redirect()
            ->to('/admin/aktivitas')
            ->with('message', 'Aktivitas berhasil dihapus.');
    }

    private function rules(): array
    {
        return [
            'title' => [
                'rules'  => 'required|string',
                'errors' => [
                    'required' => 'Judul aktivitas wajib diisi.',
                ],
            ],
            'activity_date' => [
                'rules'  => 'required|valid_date',
                'errors' => [
                    'required'   => 'Tanggal aktivitas wajib diisi.',
                    'valid_date' => 'Format tanggal tidak valid.',
                ],
            ],
            'activity_time' => [
                'rules'  => 'permit_empty|regex_match[/^([01]\\d|2[0-3]):[0-5]\\d(:[0-5]\\d)?$/]',
                'errors' => [
                    'regex_match' => 'Format waktu harus HH:MM atau HH:MM:SS.',
                ],
            ],
            'location' => [
                'rules' => 'permit_empty|string',
            ],
            'description' => [
                'rules' => 'permit_empty|string',
            ],
            'media' => [
                'rules'  => 'permit_empty|mime_in[media,image/jpg,image/jpeg,image/png,image/webp,video/mp4,video/quicktime,video/x-msvideo,video/x-matroska]|max_size[media,10240]',
                'errors' => [
                    'mime_in'  => 'Format media harus JPG, JPEG, PNG, WEBP, atau MP4.',
                    'max_size' => 'Ukuran media maksimal 10 MB.',
                ],
            ],
        ];
    }

    private function storeActivityMedia(?UploadedFile $mediaFile): array
    {
        if (! $mediaFile instanceof UploadedFile) {
            return [];
        }

        if ($mediaFile->getError() === UPLOAD_ERR_NO_FILE) {
            return [];
        }

        if (! $mediaFile->isValid() || $mediaFile->hasMoved()) {
            log_message('error', 'Upload media aktivitas tidak valid: {error}', [
                'error' => $mediaFile->getErrorString(),
            ]);

            return [];
        }

        $mime = (string) $mediaFile->getMimeType();
        $type = strpos($mime, 'video') === 0 ? 'video' : 'image';

        $targetDir = FCPATH . 'uploads/activities';
        if (! is_dir($targetDir) && ! mkdir($targetDir, 0755, true) && ! is_dir($targetDir)) {
            log_message('error', 'Gagal membuat direktori media aktivitas: {dir}', ['dir' => $targetDir]);
            return [];
        }

        $newName = $mediaFile->getRandomName();

        try {
            $mediaFile->move($targetDir, $newName);
        } catch (\Throwable $th) {
            log_message('error', 'Gagal memindahkan media aktivitas: {message}', ['message' => $th->getMessage()]);
            return [];
        }

        return [
            'path' => 'uploads/activities/' . $newName,
            'type' => $type,
        ];
    }

    private function deleteActivityMedia(?string $relativePath): void
    {
        if (! $relativePath) {
            return;
        }

        $fullPath = FCPATH . ltrim(str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $relativePath), DIRECTORY_SEPARATOR);
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    private function findOrFail(int $id): array
    {
        $activity = $this->activities->find($id);
        if (! $activity) {
            throw PageNotFoundException::forPageNotFound();
        }

        return $activity;
    }
}
