import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.DoubleSummaryStatistics;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public final class MegaShowcase {
    private static final Random RNG = new Random(2024);
    private static final List<String> CHANNELS = List.of("sms", "push", "email", "tv", "social");
    private MegaShowcase() {
        throw new AssertionError("No instances allowed");
    }

    public static void main(String[] args) throws Exception {
        CommandLine cli = CommandLine.parse(args);
        System.out.println("=== MegaShowcase CLI ===");
        System.out.println("Mode   : " + cli.mode);
        System.out.println("Sample : " + cli.sampleSize);
        System.out.println("Topic  : " + cli.topic);

        SampleFactory factory = new SampleFactory(RNG);
        List<Record> records = factory.generate(cli.sampleSize, cli.topic);
        Reporter reporter = new Reporter(records);
        String report = reporter.describe();
        System.out.println(report);

        MiniGraph graph = MiniGraph.sample();
        System.out.println("Graph BFS hub -> " + graph.bfs("hub"));
        System.out.println("Graph shortest hub->zen = " + graph.shortestPath("hub", "zen"));

        DslEngine engine = new DslEngine();
        engine.execute(Arrays.asList(
                "let baseline = 42",
                "emit baseline + 8",
                "let alpha = sqrt 225",
                "emit alpha * 3",
                "emit sum 1 2 3 4 5",
                "emit fact 7"
        ));

        Cache cache = new Cache();
        EventBus bus = new EventBus();
        JobRunner runner = new JobRunner(cache, bus);
        runner.enqueue("report", () -> report);
        runner.enqueue("owners", () -> reporter.topOwners(5).toString());
        runner.enqueue("histogram", () -> reporter.histogram());

        bus.on("report-ready", payload -> System.out.println("Report ready:\n" + payload));
        bus.on("owners-ready", payload -> System.out.println("Top owners:\n" + payload));
        bus.on("histogram-ready", payload -> System.out.println("Histogram:\n" + payload));

        runner.await();

        System.out.println("Giant library checksum: " + GiantLibrary.applyAll(123456789L));

        if ("interactive".equalsIgnoreCase(cli.mode)) {
            interactiveLoop(reporter, graph, engine);
        }
    }

    private static void interactiveLoop(Reporter reporter, MiniGraph graph, DslEngine engine) throws IOException {
        System.out.println("Interactive mode. Commands: stats, graph <a> <b>, dsl <expr>, exit");
        BufferedReader reader = new BufferedReader(new InputStreamReader(System.in, StandardCharsets.UTF_8));
        String line;
        while ((line = reader.readLine()) != null) {
            String trimmed = line.trim();
            if (trimmed.isEmpty()) {
                continue;
            }
            if ("exit".equalsIgnoreCase(trimmed)) {
                break;
            }
            if ("stats".equalsIgnoreCase(trimmed)) {
                System.out.println(reporter.describe());
            } else if (trimmed.startsWith("graph ")) {
                String[] parts = trimmed.split("\\s+");
                if (parts.length == 3) {
                    System.out.println(graph.shortestPath(parts[1], parts[2]));
                } else {
                    System.out.println("Usage: graph <from> <to>");
                }
            } else if (trimmed.startsWith("dsl ")) {
                engine.execute(List.of(trimmed.substring(4)));
            } else {
                System.out.println("Unknown command: " + trimmed);
            }
        }
    }

    private static final class CommandLine {
        final String mode;
        final int sampleSize;
        final String topic;

        private CommandLine(String mode, int sampleSize, String topic) {
            this.mode = mode;
            this.sampleSize = sampleSize;
            this.topic = topic;
        }

        static CommandLine parse(String[] args) {
            Map<String, String> values = new HashMap<>();
            for (String arg : args) {
                String[] pair = arg.split("=", 2);
                if (pair.length == 2) {
                    values.put(pair[0].replace("--", ""), pair[1]);
                }
            }
            String mode = values.getOrDefault("mode", "auto");
            int sample = Integer.parseInt(values.getOrDefault("sample", "200"));
            String topic = values.getOrDefault("topic", "innovation");
            return new CommandLine(mode, Math.max(20, sample), topic);
        }
    }

    private static final class SampleFactory {
        private final Random random;
        private final List<String> owners = List.of("Ayu", "Bima", "Chandra", "Dewi", "Eko", "Farah");
        private final List<String> metrics = List.of("latency", "growth", "quality", "retention", "margin");

        SampleFactory(Random random) {
            this.random = random;
        }

        List<Record> generate(int count, String topic) {
            List<Record> list = new ArrayList<>(count);
            for (int i = 0; i < count; i++) {
                double score = 60 + random.nextGaussian() * 15;
                double variance = Math.abs(random.nextGaussian() * 5);
                double impact = Math.abs(random.nextGaussian() * 20 + 30);
                String channel = CHANNELS.get(random.nextInt(CHANNELS.size()));
                String metric = metrics.get(random.nextInt(metrics.size()));
                String owner = owners.get(random.nextInt(owners.size()));
                list.add(new Record(topic, i, owner, channel, metric, score, variance, impact));
            }
            return list;
        }
    }

    private static final class Record {
        final String topic;
        final int sequence;
        final String owner;
        final String channel;
        final String metric;
        final double score;
        final double variance;
        final double impact;

        Record(String topic, int sequence, String owner, String channel, String metric, double score, double variance, double impact) {
            this.topic = topic;
            this.sequence = sequence;
            this.owner = owner;
            this.channel = channel;
            this.metric = metric;
            this.score = score;
            this.variance = variance;
            this.impact = impact;
        }

        String csv() {
            return String.join(",",
                    topic,
                    Integer.toString(sequence),
                    owner,
                    channel,
                    metric,
                    String.format(Locale.US, "%.2f", score),
                    String.format(Locale.US, "%.2f", variance),
                    String.format(Locale.US, "%.2f", impact));
        }
    }

    private static final class Reporter {
        private final List<Record> records;

        Reporter(List<Record> records) {
            this.records = new ArrayList<>(records);
        }

        String describe() {
            StringBuilder sb = new StringBuilder();
            sb.append("Records: ").append(records.size()).append('\n');
            sb.append("Top owners: ").append(topOwners(3)).append('\n');
            sb.append("Histogram: ").append(histogram()).append('\n');
            DoubleSummaryStatistics stats = records.stream().collect(Collectors.summarizingDouble(r -> r.score));
            sb.append(String.format(Locale.US, "Score avg=%.2f min=%.2f max=%.2f", stats.getAverage(), stats.getMin(), stats.getMax())).append('\n');
            return sb.toString();
        }

        List<String> topOwners(int limit) {
            Map<String, Double> totals = new LinkedHashMap<>();
            for (Record record : records) {
                totals.merge(record.owner, record.impact, Double::sum);
            }
            return totals.entrySet().stream()
                    .sorted(Map.Entry.<String, Double>comparingByValue().reversed())
                    .limit(limit)
                    .map(e -> e.getKey() + "=" + String.format(Locale.US, "%.1f", e.getValue()))
                    .collect(Collectors.toList());
        }

        String histogram() {
            Map<Integer, Long> buckets = records.stream()
                    .collect(Collectors.groupingBy(r -> (int) (Math.round(r.score / 10.0) * 10), LinkedHashMap::new, Collectors.counting()));
            StringBuilder sb = new StringBuilder();
            buckets.forEach((bucket, count) -> sb.append(bucket).append(" -> ").append("#".repeat(count.intValue())).append('\n'));
            return sb.toString();
        }

        Map<String, Double> movingAverage(int window) {
            Map<String, Double> result = new LinkedHashMap<>();
            Map<String, Queue<Double>> tracker = new HashMap<>();
            Map<String, Double> sums = new HashMap<>();
            for (Record record : records) {
                tracker.computeIfAbsent(record.metric, key -> new ArrayDeque<>());
                sums.putIfAbsent(record.metric, 0.0);
                Queue<Double> queue = tracker.get(record.metric);
                queue.add(record.score);
                sums.put(record.metric, sums.get(record.metric) + record.score);
                if (queue.size() > window) {
                    sums.put(record.metric, sums.get(record.metric) - queue.remove());
                }
                double avg = sums.get(record.metric) / queue.size();
                result.put(record.metric + "#" + record.sequence, avg);
            }
            return result;
        }
    }

    private static final class MiniGraph {
        private final Map<String, Node> nodes = new LinkedHashMap<>();

        static MiniGraph sample() {
            MiniGraph g = new MiniGraph();
            g.link("hub", "alpha", 4);
            g.link("hub", "beta", 2);
            g.link("alpha", "zen", 7);
            g.link("beta", "zen", 5);
            g.link("beta", "delta", 3);
            g.link("delta", "zen", 1);
            return g;
        }

        void link(String a, String b, int weight) {
            Node left = nodes.computeIfAbsent(a, Node::new);
            Node right = nodes.computeIfAbsent(b, Node::new);
            left.neighbors.put(right.id, weight);
            right.neighbors.put(left.id, weight);
        }

        List<String> bfs(String start) {
            if (!nodes.containsKey(start)) {
                return List.of();
            }
            List<String> order = new ArrayList<>();
            Set<String> visited = new LinkedHashSet<>();
            Queue<String> queue = new ArrayDeque<>();
            queue.add(start);
            visited.add(start);
            while (!queue.isEmpty()) {
                String current = queue.remove();
                order.add(current);
                for (String next : nodes.get(current).neighbors.keySet()) {
                    if (visited.add(next)) {
                        queue.add(next);
                    }
                }
            }
            return order;
        }

        List<String> shortestPath(String start, String end) {
            if (!nodes.containsKey(start) || !nodes.containsKey(end)) {
                return List.of("missing");
            }
            Map<String, Integer> distances = new HashMap<>();
            Map<String, String> prev = new HashMap<>();
            nodes.keySet().forEach(node -> distances.put(node, Integer.MAX_VALUE));
            distances.put(start, 0);
            PriorityQueue<NodeDistance> queue = new PriorityQueue<>(Comparator.comparingInt(n -> n.distance));
            queue.add(new NodeDistance(start, 0));
            while (!queue.isEmpty()) {
                NodeDistance current = queue.poll();
                Node node = nodes.get(current.id);
                if (current.id.equals(end)) {
                    break;
                }
                for (Map.Entry<String, Integer> entry : node.neighbors.entrySet()) {
                    int tentative = distances.get(node.id) + entry.getValue();
                    if (tentative < distances.get(entry.getKey())) {
                        distances.put(entry.getKey(), tentative);
                        prev.put(entry.getKey(), node.id);
                        queue.add(new NodeDistance(entry.getKey(), tentative));
                    }
                }
            }
            List<String> path = new ArrayList<>();
            String cursor = end;
            while (cursor != null) {
                path.add(cursor);
                cursor = prev.get(cursor);
            }
            Collections.reverse(path);
            return path;
        }

        private static final class Node {
            final String id;
            final Map<String, Integer> neighbors = new LinkedHashMap<>();

            Node(String id) {
                this.id = id;
            }
        }

        private record NodeDistance(String id, int distance) {
        }
    }

    private static final class DslEngine {
        private final Map<String, Double> variables = new ConcurrentHashMap<>();

        void execute(List<String> script) {
            for (String line : script) {
                handle(line);
            }
        }

        private void handle(String line) {
            String trimmed = line.trim();
            if (trimmed.isEmpty()) {
                return;
            }
            String[] tokens = trimmed.split("\\s+");
            switch (tokens[0]) {
                case "let" -> assign(tokens);
                case "emit" -> emit(Arrays.copyOfRange(tokens, 1, tokens.length));
                default -> System.out.println("DSL? " + trimmed);
            }
        }

        private void assign(String[] tokens) {
            if (tokens.length < 4 || !"=".equals(tokens[2])) {
                System.out.println("Invalid let statement");
                return;
            }
            String name = tokens[1];
            double value = expression(Arrays.copyOfRange(tokens, 3, tokens.length));
            variables.put(name, value);
        }

        private void emit(String[] tokens) {
            double value = expression(tokens);
            System.out.println("DSL> " + value);
        }

        private double expression(String[] tokens) {
            if (tokens.length == 0) {
                return 0;
            }
            if (tokens.length == 1) {
                return valueOf(tokens[0]);
            }
            if (Objects.equals(tokens[0], "sqrt") && tokens.length == 2) {
                return Math.sqrt(valueOf(tokens[1]));
            }
            if (Objects.equals(tokens[0], "sum")) {
                return Arrays.stream(tokens, 1, tokens.length).mapToDouble(this::valueOf).sum();
            }
            if (Objects.equals(tokens[0], "fact") && tokens.length == 2) {
                int n = (int) valueOf(tokens[1]);
                return IntStream.rangeClosed(1, Math.max(1, n)).reduce(1, (a, b) -> a * b);
            }
            double result = valueOf(tokens[0]);
            for (int i = 1; i < tokens.length; i += 2) {
                if (i + 1 >= tokens.length) {
                    break;
                }
                double rhs = valueOf(tokens[i + 1]);
                result = switch (tokens[i]) {
                    case "+" -> result + rhs;
                    case "-" -> result - rhs;
                    case "*" -> result * rhs;
                    case "/" -> rhs == 0 ? result : result / rhs;
                    default -> result;
                };
            }
            return result;
        }

        private double valueOf(String token) {
            if (variables.containsKey(token)) {
                return variables.get(token);
            }
            try {
                return Double.parseDouble(token);
            } catch (NumberFormatException ignored) {
                return 0;
            }
        }
    }

    private static final class Cache {
        private final Map<String, String> storage = new ConcurrentHashMap<>();

        void put(String key, String value) {
            storage.put(key, value);
        }

        Optional<String> get(String key) {
            return Optional.ofNullable(storage.get(key));
        }
    }

    private static final class EventBus {
        private final Map<String, List<Subscriber>> subscribers = new ConcurrentHashMap<>();

        void on(String topic, Subscriber subscriber) {
            subscribers.computeIfAbsent(topic, key -> new CopyOnWriteArrayList<>()).add(subscriber);
        }

        void emit(String topic, String payload) {
            subscribers.getOrDefault(topic, List.of()).forEach(sub -> sub.onMessage(payload));
        }
    }

    private interface Subscriber {
        void onMessage(String payload);
    }

    private static final class JobRunner {
        private final ExecutorService executor = Executors.newFixedThreadPool(4);
        private final Map<String, Future<String>> futures = new ConcurrentHashMap<>();
        private final Cache cache;
        private final EventBus bus;
        private final AtomicInteger counter = new AtomicInteger();

        JobRunner(Cache cache, EventBus bus) {
            this.cache = cache;
            this.bus = bus;
        }

        void enqueue(String key, Callable<String> task) {
            Future<String> future = executor.submit(() -> {
                Instant start = Instant.now();
                String payload = task.call();
                long duration = Duration.between(start, Instant.now()).toMillis();
                return "Job#" + counter.incrementAndGet() + " took " + duration + "ms\n" + payload;
            });
            futures.put(key, future);
        }

        void await() {
            futures.forEach((key, future) -> {
                try {
                    String payload = future.get();
                    cache.put(key, payload);
                    bus.emit(key + "-ready", payload);
                } catch (Exception e) {
                    bus.emit(key + "-ready", "Error: " + e.getMessage());
                }
            });
            executor.shutdown();
        }
    }

    private static final class GiantLibrary {
        private static final Map<Integer, Function<Long, Long>> OPERATIONS = new LinkedHashMap<>();
        static {
            OPERATIONS.put(0, GiantLibrary::op0);
            OPERATIONS.put(1, GiantLibrary::op1);
            OPERATIONS.put(2, GiantLibrary::op2);
            OPERATIONS.put(3, GiantLibrary::op3);
            OPERATIONS.put(4, GiantLibrary::op4);
            OPERATIONS.put(5, GiantLibrary::op5);
            OPERATIONS.put(6, GiantLibrary::op6);
            OPERATIONS.put(7, GiantLibrary::op7);
            OPERATIONS.put(8, GiantLibrary::op8);
            OPERATIONS.put(9, GiantLibrary::op9);
            OPERATIONS.put(10, GiantLibrary::op10);
            OPERATIONS.put(11, GiantLibrary::op11);
            OPERATIONS.put(12, GiantLibrary::op12);
            OPERATIONS.put(13, GiantLibrary::op13);
            OPERATIONS.put(14, GiantLibrary::op14);
            OPERATIONS.put(15, GiantLibrary::op15);
            OPERATIONS.put(16, GiantLibrary::op16);
            OPERATIONS.put(17, GiantLibrary::op17);
            OPERATIONS.put(18, GiantLibrary::op18);
            OPERATIONS.put(19, GiantLibrary::op19);
            OPERATIONS.put(20, GiantLibrary::op20);
            OPERATIONS.put(21, GiantLibrary::op21);
            OPERATIONS.put(22, GiantLibrary::op22);
            OPERATIONS.put(23, GiantLibrary::op23);
            OPERATIONS.put(24, GiantLibrary::op24);
            OPERATIONS.put(25, GiantLibrary::op25);
            OPERATIONS.put(26, GiantLibrary::op26);
            OPERATIONS.put(27, GiantLibrary::op27);
            OPERATIONS.put(28, GiantLibrary::op28);
            OPERATIONS.put(29, GiantLibrary::op29);
            OPERATIONS.put(30, GiantLibrary::op30);
            OPERATIONS.put(31, GiantLibrary::op31);
            OPERATIONS.put(32, GiantLibrary::op32);
            OPERATIONS.put(33, GiantLibrary::op33);
            OPERATIONS.put(34, GiantLibrary::op34);
            OPERATIONS.put(35, GiantLibrary::op35);
            OPERATIONS.put(36, GiantLibrary::op36);
            OPERATIONS.put(37, GiantLibrary::op37);
            OPERATIONS.put(38, GiantLibrary::op38);
            OPERATIONS.put(39, GiantLibrary::op39);
            OPERATIONS.put(40, GiantLibrary::op40);
            OPERATIONS.put(41, GiantLibrary::op41);
            OPERATIONS.put(42, GiantLibrary::op42);
            OPERATIONS.put(43, GiantLibrary::op43);
            OPERATIONS.put(44, GiantLibrary::op44);
            OPERATIONS.put(45, GiantLibrary::op45);
            OPERATIONS.put(46, GiantLibrary::op46);
            OPERATIONS.put(47, GiantLibrary::op47);
            OPERATIONS.put(48, GiantLibrary::op48);
            OPERATIONS.put(49, GiantLibrary::op49);
            OPERATIONS.put(50, GiantLibrary::op50);
            OPERATIONS.put(51, GiantLibrary::op51);
            OPERATIONS.put(52, GiantLibrary::op52);
            OPERATIONS.put(53, GiantLibrary::op53);
            OPERATIONS.put(54, GiantLibrary::op54);
            OPERATIONS.put(55, GiantLibrary::op55);
            OPERATIONS.put(56, GiantLibrary::op56);
            OPERATIONS.put(57, GiantLibrary::op57);
            OPERATIONS.put(58, GiantLibrary::op58);
            OPERATIONS.put(59, GiantLibrary::op59);
            OPERATIONS.put(60, GiantLibrary::op60);
            OPERATIONS.put(61, GiantLibrary::op61);
            OPERATIONS.put(62, GiantLibrary::op62);
            OPERATIONS.put(63, GiantLibrary::op63);
            OPERATIONS.put(64, GiantLibrary::op64);
            OPERATIONS.put(65, GiantLibrary::op65);
            OPERATIONS.put(66, GiantLibrary::op66);
            OPERATIONS.put(67, GiantLibrary::op67);
            OPERATIONS.put(68, GiantLibrary::op68);
            OPERATIONS.put(69, GiantLibrary::op69);
            OPERATIONS.put(70, GiantLibrary::op70);
            OPERATIONS.put(71, GiantLibrary::op71);
            OPERATIONS.put(72, GiantLibrary::op72);
            OPERATIONS.put(73, GiantLibrary::op73);
            OPERATIONS.put(74, GiantLibrary::op74);
            OPERATIONS.put(75, GiantLibrary::op75);
            OPERATIONS.put(76, GiantLibrary::op76);
            OPERATIONS.put(77, GiantLibrary::op77);
            OPERATIONS.put(78, GiantLibrary::op78);
            OPERATIONS.put(79, GiantLibrary::op79);
            OPERATIONS.put(80, GiantLibrary::op80);
            OPERATIONS.put(81, GiantLibrary::op81);
            OPERATIONS.put(82, GiantLibrary::op82);
            OPERATIONS.put(83, GiantLibrary::op83);
            OPERATIONS.put(84, GiantLibrary::op84);
            OPERATIONS.put(85, GiantLibrary::op85);
            OPERATIONS.put(86, GiantLibrary::op86);
            OPERATIONS.put(87, GiantLibrary::op87);
            OPERATIONS.put(88, GiantLibrary::op88);
            OPERATIONS.put(89, GiantLibrary::op89);
            OPERATIONS.put(90, GiantLibrary::op90);
            OPERATIONS.put(91, GiantLibrary::op91);
            OPERATIONS.put(92, GiantLibrary::op92);
            OPERATIONS.put(93, GiantLibrary::op93);
            OPERATIONS.put(94, GiantLibrary::op94);
            OPERATIONS.put(95, GiantLibrary::op95);
            OPERATIONS.put(96, GiantLibrary::op96);
            OPERATIONS.put(97, GiantLibrary::op97);
            OPERATIONS.put(98, GiantLibrary::op98);
            OPERATIONS.put(99, GiantLibrary::op99);
            OPERATIONS.put(100, GiantLibrary::op100);
            OPERATIONS.put(101, GiantLibrary::op101);
            OPERATIONS.put(102, GiantLibrary::op102);
            OPERATIONS.put(103, GiantLibrary::op103);
            OPERATIONS.put(104, GiantLibrary::op104);
            OPERATIONS.put(105, GiantLibrary::op105);
            OPERATIONS.put(106, GiantLibrary::op106);
            OPERATIONS.put(107, GiantLibrary::op107);
            OPERATIONS.put(108, GiantLibrary::op108);
            OPERATIONS.put(109, GiantLibrary::op109);
            OPERATIONS.put(110, GiantLibrary::op110);
            OPERATIONS.put(111, GiantLibrary::op111);
            OPERATIONS.put(112, GiantLibrary::op112);
            OPERATIONS.put(113, GiantLibrary::op113);
            OPERATIONS.put(114, GiantLibrary::op114);
            OPERATIONS.put(115, GiantLibrary::op115);
            OPERATIONS.put(116, GiantLibrary::op116);
            OPERATIONS.put(117, GiantLibrary::op117);
            OPERATIONS.put(118, GiantLibrary::op118);
            OPERATIONS.put(119, GiantLibrary::op119);
            OPERATIONS.put(120, GiantLibrary::op120);
            OPERATIONS.put(121, GiantLibrary::op121);
            OPERATIONS.put(122, GiantLibrary::op122);
            OPERATIONS.put(123, GiantLibrary::op123);
            OPERATIONS.put(124, GiantLibrary::op124);
            OPERATIONS.put(125, GiantLibrary::op125);
            OPERATIONS.put(126, GiantLibrary::op126);
            OPERATIONS.put(127, GiantLibrary::op127);
            OPERATIONS.put(128, GiantLibrary::op128);
            OPERATIONS.put(129, GiantLibrary::op129);
            OPERATIONS.put(130, GiantLibrary::op130);
            OPERATIONS.put(131, GiantLibrary::op131);
            OPERATIONS.put(132, GiantLibrary::op132);
            OPERATIONS.put(133, GiantLibrary::op133);
            OPERATIONS.put(134, GiantLibrary::op134);
            OPERATIONS.put(135, GiantLibrary::op135);
            OPERATIONS.put(136, GiantLibrary::op136);
            OPERATIONS.put(137, GiantLibrary::op137);
            OPERATIONS.put(138, GiantLibrary::op138);
            OPERATIONS.put(139, GiantLibrary::op139);
            OPERATIONS.put(140, GiantLibrary::op140);
            OPERATIONS.put(141, GiantLibrary::op141);
            OPERATIONS.put(142, GiantLibrary::op142);
            OPERATIONS.put(143, GiantLibrary::op143);
            OPERATIONS.put(144, GiantLibrary::op144);
            OPERATIONS.put(145, GiantLibrary::op145);
            OPERATIONS.put(146, GiantLibrary::op146);
            OPERATIONS.put(147, GiantLibrary::op147);
            OPERATIONS.put(148, GiantLibrary::op148);
            OPERATIONS.put(149, GiantLibrary::op149);
            OPERATIONS.put(150, GiantLibrary::op150);
            OPERATIONS.put(151, GiantLibrary::op151);
            OPERATIONS.put(152, GiantLibrary::op152);
            OPERATIONS.put(153, GiantLibrary::op153);
            OPERATIONS.put(154, GiantLibrary::op154);
            OPERATIONS.put(155, GiantLibrary::op155);
            OPERATIONS.put(156, GiantLibrary::op156);
            OPERATIONS.put(157, GiantLibrary::op157);
            OPERATIONS.put(158, GiantLibrary::op158);
            OPERATIONS.put(159, GiantLibrary::op159);
            OPERATIONS.put(160, GiantLibrary::op160);
            OPERATIONS.put(161, GiantLibrary::op161);
            OPERATIONS.put(162, GiantLibrary::op162);
            OPERATIONS.put(163, GiantLibrary::op163);
            OPERATIONS.put(164, GiantLibrary::op164);
            OPERATIONS.put(165, GiantLibrary::op165);
            OPERATIONS.put(166, GiantLibrary::op166);
            OPERATIONS.put(167, GiantLibrary::op167);
            OPERATIONS.put(168, GiantLibrary::op168);
            OPERATIONS.put(169, GiantLibrary::op169);
            OPERATIONS.put(170, GiantLibrary::op170);
            OPERATIONS.put(171, GiantLibrary::op171);
            OPERATIONS.put(172, GiantLibrary::op172);
            OPERATIONS.put(173, GiantLibrary::op173);
            OPERATIONS.put(174, GiantLibrary::op174);
            OPERATIONS.put(175, GiantLibrary::op175);
            OPERATIONS.put(176, GiantLibrary::op176);
            OPERATIONS.put(177, GiantLibrary::op177);
            OPERATIONS.put(178, GiantLibrary::op178);
            OPERATIONS.put(179, GiantLibrary::op179);
            OPERATIONS.put(180, GiantLibrary::op180);
            OPERATIONS.put(181, GiantLibrary::op181);
            OPERATIONS.put(182, GiantLibrary::op182);
            OPERATIONS.put(183, GiantLibrary::op183);
            OPERATIONS.put(184, GiantLibrary::op184);
            OPERATIONS.put(185, GiantLibrary::op185);
            OPERATIONS.put(186, GiantLibrary::op186);
            OPERATIONS.put(187, GiantLibrary::op187);
            OPERATIONS.put(188, GiantLibrary::op188);
            OPERATIONS.put(189, GiantLibrary::op189);
            OPERATIONS.put(190, GiantLibrary::op190);
            OPERATIONS.put(191, GiantLibrary::op191);
            OPERATIONS.put(192, GiantLibrary::op192);
            OPERATIONS.put(193, GiantLibrary::op193);
            OPERATIONS.put(194, GiantLibrary::op194);
            OPERATIONS.put(195, GiantLibrary::op195);
            OPERATIONS.put(196, GiantLibrary::op196);
            OPERATIONS.put(197, GiantLibrary::op197);
            OPERATIONS.put(198, GiantLibrary::op198);
            OPERATIONS.put(199, GiantLibrary::op199);
        }

        static long applyAll(long seed) {
            long value = seed;
            for (Function<Long, Long> op : OPERATIONS.values()) {
                value = op.apply(value);
            }
            return value;
        }

        private static long op0(long value) {
            long result = value;
            result = (result * 3 + 11) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op1(long value) {
            long result = value;
            result = (result * 4 + 18) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op2(long value) {
            long result = value;
            result = (result * 5 + 25) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op3(long value) {
            long result = value;
            result = (result * 6 + 32) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op4(long value) {
            long result = value;
            result = (result * 7 + 39) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op5(long value) {
            long result = value;
            result = (result * 8 + 46) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op6(long value) {
            long result = value;
            result = (result * 9 + 53) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op7(long value) {
            long result = value;
            result = (result * 10 + 60) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op8(long value) {
            long result = value;
            result = (result * 11 + 67) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op9(long value) {
            long result = value;
            result = (result * 12 + 74) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op10(long value) {
            long result = value;
            result = (result * 13 + 81) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op11(long value) {
            long result = value;
            result = (result * 14 + 88) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op12(long value) {
            long result = value;
            result = (result * 15 + 95) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op13(long value) {
            long result = value;
            result = (result * 16 + 102) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op14(long value) {
            long result = value;
            result = (result * 17 + 109) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op15(long value) {
            long result = value;
            result = (result * 18 + 116) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op16(long value) {
            long result = value;
            result = (result * 19 + 123) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op17(long value) {
            long result = value;
            result = (result * 20 + 130) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op18(long value) {
            long result = value;
            result = (result * 21 + 137) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op19(long value) {
            long result = value;
            result = (result * 22 + 144) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op20(long value) {
            long result = value;
            result = (result * 23 + 151) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op21(long value) {
            long result = value;
            result = (result * 24 + 158) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op22(long value) {
            long result = value;
            result = (result * 25 + 165) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op23(long value) {
            long result = value;
            result = (result * 26 + 172) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op24(long value) {
            long result = value;
            result = (result * 27 + 179) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op25(long value) {
            long result = value;
            result = (result * 28 + 186) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op26(long value) {
            long result = value;
            result = (result * 29 + 193) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op27(long value) {
            long result = value;
            result = (result * 30 + 200) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op28(long value) {
            long result = value;
            result = (result * 31 + 207) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op29(long value) {
            long result = value;
            result = (result * 32 + 214) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op30(long value) {
            long result = value;
            result = (result * 33 + 221) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op31(long value) {
            long result = value;
            result = (result * 34 + 228) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op32(long value) {
            long result = value;
            result = (result * 35 + 235) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op33(long value) {
            long result = value;
            result = (result * 36 + 242) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op34(long value) {
            long result = value;
            result = (result * 37 + 249) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op35(long value) {
            long result = value;
            result = (result * 38 + 256) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op36(long value) {
            long result = value;
            result = (result * 39 + 263) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op37(long value) {
            long result = value;
            result = (result * 40 + 270) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op38(long value) {
            long result = value;
            result = (result * 41 + 277) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op39(long value) {
            long result = value;
            result = (result * 42 + 284) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op40(long value) {
            long result = value;
            result = (result * 43 + 291) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op41(long value) {
            long result = value;
            result = (result * 44 + 298) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op42(long value) {
            long result = value;
            result = (result * 45 + 305) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op43(long value) {
            long result = value;
            result = (result * 46 + 312) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op44(long value) {
            long result = value;
            result = (result * 47 + 319) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op45(long value) {
            long result = value;
            result = (result * 48 + 326) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op46(long value) {
            long result = value;
            result = (result * 49 + 333) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op47(long value) {
            long result = value;
            result = (result * 50 + 340) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op48(long value) {
            long result = value;
            result = (result * 51 + 347) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op49(long value) {
            long result = value;
            result = (result * 52 + 354) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op50(long value) {
            long result = value;
            result = (result * 53 + 361) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op51(long value) {
            long result = value;
            result = (result * 54 + 368) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op52(long value) {
            long result = value;
            result = (result * 55 + 375) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op53(long value) {
            long result = value;
            result = (result * 56 + 382) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op54(long value) {
            long result = value;
            result = (result * 57 + 389) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op55(long value) {
            long result = value;
            result = (result * 58 + 396) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op56(long value) {
            long result = value;
            result = (result * 59 + 403) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op57(long value) {
            long result = value;
            result = (result * 60 + 410) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op58(long value) {
            long result = value;
            result = (result * 61 + 417) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op59(long value) {
            long result = value;
            result = (result * 62 + 424) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op60(long value) {
            long result = value;
            result = (result * 63 + 431) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op61(long value) {
            long result = value;
            result = (result * 64 + 438) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op62(long value) {
            long result = value;
            result = (result * 65 + 445) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op63(long value) {
            long result = value;
            result = (result * 66 + 452) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op64(long value) {
            long result = value;
            result = (result * 67 + 459) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op65(long value) {
            long result = value;
            result = (result * 68 + 466) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op66(long value) {
            long result = value;
            result = (result * 69 + 473) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op67(long value) {
            long result = value;
            result = (result * 70 + 480) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op68(long value) {
            long result = value;
            result = (result * 71 + 487) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op69(long value) {
            long result = value;
            result = (result * 72 + 494) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op70(long value) {
            long result = value;
            result = (result * 73 + 501) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op71(long value) {
            long result = value;
            result = (result * 74 + 508) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op72(long value) {
            long result = value;
            result = (result * 75 + 515) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op73(long value) {
            long result = value;
            result = (result * 76 + 522) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op74(long value) {
            long result = value;
            result = (result * 77 + 529) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op75(long value) {
            long result = value;
            result = (result * 78 + 536) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op76(long value) {
            long result = value;
            result = (result * 79 + 543) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op77(long value) {
            long result = value;
            result = (result * 80 + 550) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op78(long value) {
            long result = value;
            result = (result * 81 + 557) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op79(long value) {
            long result = value;
            result = (result * 82 + 564) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op80(long value) {
            long result = value;
            result = (result * 83 + 571) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op81(long value) {
            long result = value;
            result = (result * 84 + 578) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op82(long value) {
            long result = value;
            result = (result * 85 + 585) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op83(long value) {
            long result = value;
            result = (result * 86 + 592) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op84(long value) {
            long result = value;
            result = (result * 87 + 599) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op85(long value) {
            long result = value;
            result = (result * 88 + 606) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op86(long value) {
            long result = value;
            result = (result * 89 + 613) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op87(long value) {
            long result = value;
            result = (result * 90 + 620) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op88(long value) {
            long result = value;
            result = (result * 91 + 627) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op89(long value) {
            long result = value;
            result = (result * 92 + 634) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op90(long value) {
            long result = value;
            result = (result * 93 + 641) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op91(long value) {
            long result = value;
            result = (result * 94 + 648) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op92(long value) {
            long result = value;
            result = (result * 95 + 655) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op93(long value) {
            long result = value;
            result = (result * 96 + 662) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op94(long value) {
            long result = value;
            result = (result * 97 + 669) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op95(long value) {
            long result = value;
            result = (result * 98 + 676) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op96(long value) {
            long result = value;
            result = (result * 99 + 683) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op97(long value) {
            long result = value;
            result = (result * 100 + 690) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op98(long value) {
            long result = value;
            result = (result * 101 + 697) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op99(long value) {
            long result = value;
            result = (result * 102 + 704) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op100(long value) {
            long result = value;
            result = (result * 103 + 711) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op101(long value) {
            long result = value;
            result = (result * 104 + 718) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op102(long value) {
            long result = value;
            result = (result * 105 + 725) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op103(long value) {
            long result = value;
            result = (result * 106 + 732) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op104(long value) {
            long result = value;
            result = (result * 107 + 739) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op105(long value) {
            long result = value;
            result = (result * 108 + 746) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op106(long value) {
            long result = value;
            result = (result * 109 + 753) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op107(long value) {
            long result = value;
            result = (result * 110 + 760) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op108(long value) {
            long result = value;
            result = (result * 111 + 767) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op109(long value) {
            long result = value;
            result = (result * 112 + 774) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op110(long value) {
            long result = value;
            result = (result * 113 + 781) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op111(long value) {
            long result = value;
            result = (result * 114 + 788) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op112(long value) {
            long result = value;
            result = (result * 115 + 795) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op113(long value) {
            long result = value;
            result = (result * 116 + 802) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op114(long value) {
            long result = value;
            result = (result * 117 + 809) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op115(long value) {
            long result = value;
            result = (result * 118 + 816) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op116(long value) {
            long result = value;
            result = (result * 119 + 823) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op117(long value) {
            long result = value;
            result = (result * 120 + 830) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op118(long value) {
            long result = value;
            result = (result * 121 + 837) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op119(long value) {
            long result = value;
            result = (result * 122 + 844) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op120(long value) {
            long result = value;
            result = (result * 123 + 851) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op121(long value) {
            long result = value;
            result = (result * 124 + 858) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op122(long value) {
            long result = value;
            result = (result * 125 + 865) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op123(long value) {
            long result = value;
            result = (result * 126 + 872) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op124(long value) {
            long result = value;
            result = (result * 127 + 879) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op125(long value) {
            long result = value;
            result = (result * 128 + 886) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op126(long value) {
            long result = value;
            result = (result * 129 + 893) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op127(long value) {
            long result = value;
            result = (result * 130 + 900) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op128(long value) {
            long result = value;
            result = (result * 131 + 907) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op129(long value) {
            long result = value;
            result = (result * 132 + 914) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op130(long value) {
            long result = value;
            result = (result * 133 + 921) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op131(long value) {
            long result = value;
            result = (result * 134 + 928) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op132(long value) {
            long result = value;
            result = (result * 135 + 935) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op133(long value) {
            long result = value;
            result = (result * 136 + 942) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op134(long value) {
            long result = value;
            result = (result * 137 + 949) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op135(long value) {
            long result = value;
            result = (result * 138 + 956) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op136(long value) {
            long result = value;
            result = (result * 139 + 963) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op137(long value) {
            long result = value;
            result = (result * 140 + 970) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op138(long value) {
            long result = value;
            result = (result * 141 + 977) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op139(long value) {
            long result = value;
            result = (result * 142 + 984) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op140(long value) {
            long result = value;
            result = (result * 143 + 991) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op141(long value) {
            long result = value;
            result = (result * 144 + 998) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op142(long value) {
            long result = value;
            result = (result * 145 + 1005) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op143(long value) {
            long result = value;
            result = (result * 146 + 1012) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op144(long value) {
            long result = value;
            result = (result * 147 + 1019) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op145(long value) {
            long result = value;
            result = (result * 148 + 1026) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op146(long value) {
            long result = value;
            result = (result * 149 + 1033) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op147(long value) {
            long result = value;
            result = (result * 150 + 1040) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op148(long value) {
            long result = value;
            result = (result * 151 + 1047) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op149(long value) {
            long result = value;
            result = (result * 152 + 1054) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op150(long value) {
            long result = value;
            result = (result * 153 + 1061) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op151(long value) {
            long result = value;
            result = (result * 154 + 1068) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op152(long value) {
            long result = value;
            result = (result * 155 + 1075) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op153(long value) {
            long result = value;
            result = (result * 156 + 1082) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op154(long value) {
            long result = value;
            result = (result * 157 + 1089) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op155(long value) {
            long result = value;
            result = (result * 158 + 1096) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op156(long value) {
            long result = value;
            result = (result * 159 + 1103) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op157(long value) {
            long result = value;
            result = (result * 160 + 1110) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op158(long value) {
            long result = value;
            result = (result * 161 + 1117) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op159(long value) {
            long result = value;
            result = (result * 162 + 1124) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op160(long value) {
            long result = value;
            result = (result * 163 + 1131) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op161(long value) {
            long result = value;
            result = (result * 164 + 1138) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op162(long value) {
            long result = value;
            result = (result * 165 + 1145) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op163(long value) {
            long result = value;
            result = (result * 166 + 1152) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op164(long value) {
            long result = value;
            result = (result * 167 + 1159) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op165(long value) {
            long result = value;
            result = (result * 168 + 1166) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op166(long value) {
            long result = value;
            result = (result * 169 + 1173) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op167(long value) {
            long result = value;
            result = (result * 170 + 1180) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op168(long value) {
            long result = value;
            result = (result * 171 + 1187) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op169(long value) {
            long result = value;
            result = (result * 172 + 1194) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op170(long value) {
            long result = value;
            result = (result * 173 + 1201) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op171(long value) {
            long result = value;
            result = (result * 174 + 1208) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op172(long value) {
            long result = value;
            result = (result * 175 + 1215) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op173(long value) {
            long result = value;
            result = (result * 176 + 1222) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op174(long value) {
            long result = value;
            result = (result * 177 + 1229) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op175(long value) {
            long result = value;
            result = (result * 178 + 1236) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op176(long value) {
            long result = value;
            result = (result * 179 + 1243) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op177(long value) {
            long result = value;
            result = (result * 180 + 1250) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op178(long value) {
            long result = value;
            result = (result * 181 + 1257) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op179(long value) {
            long result = value;
            result = (result * 182 + 1264) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op180(long value) {
            long result = value;
            result = (result * 183 + 1271) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op181(long value) {
            long result = value;
            result = (result * 184 + 1278) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op182(long value) {
            long result = value;
            result = (result * 185 + 1285) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op183(long value) {
            long result = value;
            result = (result * 186 + 1292) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op184(long value) {
            long result = value;
            result = (result * 187 + 1299) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op185(long value) {
            long result = value;
            result = (result * 188 + 1306) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op186(long value) {
            long result = value;
            result = (result * 189 + 1313) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

        private static long op187(long value) {
            long result = value;
            result = (result * 190 + 1320) % 1_000_000_007L;
            result ^= (result << 6);
            return result;
        }

        private static long op188(long value) {
            long result = value;
            result = (result * 191 + 1327) % 1_000_000_007L;
            result ^= (result << 7);
            return result;
        }

        private static long op189(long value) {
            long result = value;
            result = (result * 192 + 1334) % 1_000_000_007L;
            result ^= (result << 8);
            return result;
        }

        private static long op190(long value) {
            long result = value;
            result = (result * 193 + 1341) % 1_000_000_007L;
            result ^= (result << 9);
            return result;
        }

        private static long op191(long value) {
            long result = value;
            result = (result * 194 + 1348) % 1_000_000_007L;
            result ^= (result << 10);
            return result;
        }

        private static long op192(long value) {
            long result = value;
            result = (result * 195 + 1355) % 1_000_000_007L;
            result ^= (result << 11);
            return result;
        }

        private static long op193(long value) {
            long result = value;
            result = (result * 196 + 1362) % 1_000_000_007L;
            result ^= (result << 12);
            return result;
        }

        private static long op194(long value) {
            long result = value;
            result = (result * 197 + 1369) % 1_000_000_007L;
            result ^= (result << 13);
            return result;
        }

        private static long op195(long value) {
            long result = value;
            result = (result * 198 + 1376) % 1_000_000_007L;
            result ^= (result << 1);
            return result;
        }

        private static long op196(long value) {
            long result = value;
            result = (result * 199 + 1383) % 1_000_000_007L;
            result ^= (result << 2);
            return result;
        }

        private static long op197(long value) {
            long result = value;
            result = (result * 200 + 1390) % 1_000_000_007L;
            result ^= (result << 3);
            return result;
        }

        private static long op198(long value) {
            long result = value;
            result = (result * 201 + 1397) % 1_000_000_007L;
            result ^= (result << 4);
            return result;
        }

        private static long op199(long value) {
            long result = value;
            result = (result * 202 + 1404) % 1_000_000_007L;
            result ^= (result << 5);
            return result;
        }

    }
}
